import dayjs from 'dayjs';

export const dateUtil = {
	date(time: string, format?: string): string {
		if (!time || time.length < 10) return '';
		const date = time.substring(0, 10);
		if (format == 'yyyyMMdd') return date.replace(/-/g, '');
		if (format == 'yyMMdd') return date.substring(2).replace(/-/g, '');
		if (format == 'MM-dd') return date.substring(5);
		if (format == 'MMdd') return date.substring(5).replace(/-/g, '');
		return time.substring(0, 10);
	},
	time(time: string, hm?: boolean): string {
		if (!time || time.length < 10) return '';
		const t = time.split(' ')[1] || '00:00:00';
		if (hm) return t.substring(0, 5);
		return t;
	}
};

export function uuid(): string {
	return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
		const r = (Math.random() * 16) | 0;
		const v = c == 'x' ? r : (r & 0x3) | 0x8;
		return v.toString(16);
	});
}

export function formatHourMinute(v: string): string {
	if (!v) return '';
	v = v.replace(/(:)|(:)/g, '');
	if (v.length < 3) return v;
	v = v.length >= 4 ? v.substring(v.length - 4) : '0' + v;
	let hour = parseInt(v.substring(0, 2).replace(/^0/, '')) || 0;
	if (hour > 24) hour = 0;
	let minute = parseInt(v.substring(2, 4).replace(/^0/, '')) || 0;
	if (minute > 60) minute = 0;
	return (hour > 9 ? '' : '0') + hour + ':' + (minute > 9 ? '' : '0') + minute;
}

/**
 * 解析整则座位，格式如：11B、12D,13F、13-15B、13ABC、12-14ABC
 */
export function parseSeatNo(str: string): string[] {
	if (!str) return [];
	const list: string[] = [];
	const segs = str.split(/(,|，)/g);
	segs.forEach(seg => {
		let row = seg;
		let col = '';
		for (let i = 0; i < seg.length; i++) {
			if (/[a-z]/i.test(seg.charAt(i))) {
				row = seg.substring(0, i);
				col = seg.substring(i);
				break;
			}
		}
		const si = row.indexOf('-');
		const rowList: number[] = [];
		if (si != -1) {
			const sn = parseInt(str.substring(0, si));
			const en = parseInt(str.substring(si + 1));
			if (isNaN(sn) || isNaN(en)) return;
			for (let i = sn; i <= en; i++) rowList.push(i);
		} else {
			const rn = parseInt(row);
			if (!isNaN(rn)) rowList.push(rn);
		}
		const colList: string[] = [];
		for (let i = 0; i < col.length; i++) {
			const c = col.charAt(i);
			if (/[a-z]/i.test(c)) colList.push(c);
		}

		if (rowList.length) {
			rowList.forEach(row => {
				if (colList.length) {
					colList.forEach(col => list.push(row + '' + col));
				} else {
					list.push(row + '');
				}
			});
		} else if (colList.length) {
			colList.forEach(col => list.push('' + col));
		}
	});
	list.sort();
	return list;
}

//格式化时间: yyyy-MM-dd HH:mm
export function formatTime(v?: string): string {
	if (!v) return '';
	if (v.length == 10) v += ' 00:00';
	if (v.length <= 16) return v;
	return v.replace(/:\d{2}$/g, '');
}

//格式化日期: yyyy-MM-dd
export function formatDate(v?: string): string {
	if (!v) return '';
	if (v.length < 10) return v;
	return v.replace(/\s.+$/g, '');
}

//格式化日期: MM-dd
export function formatTimeMd(v?: string): string {
	if (!v || v.length < 10) return '';
	if (v.length == 10) return v.substring(5);
	return v.replace(/\s.+$/g, '').substring(5);
}

//格式化日期: HH:mm
export function formatTimeHm(v?: string): string {
	if (!v || v.length < 10) return '';
	if (v.length <= 16) return v.replace(/^.+\s/g, '');
	return v.replace(/(^.+\s)|(:\d{2}$)/g, '');
}

//格式化日期: MM-dd HH:mm
export function formatTimeMdHm(v?: string): string {
	if (!v || v.length < 10) return '';
	if (v.length <= 16) return v.replace(/^\d{4}-/g, '');
	return v.replace(/(^\d{4}-)|(:\d{2}$)/g, '');
}

//获取查询时间端
export function getDateBetween(days: number): string[] {
	const now = dayjs();
	return [now.add(-days, 'day').format('YYYY-MM-DD') + ' 00:00:00', now.add(1, 'day').format('YYYY-MM-DD') + ' 00:00:00'];
}

//不在线消息判断
export function isOfflineMsg(str?: string): boolean {
	if (!str) return false;
	if (str.indexOf('android.database.sqlite.SQLiteException') != -1) return false;
	return /\srequest|\sfail|\sconnect|\shost|\stimeout/gi.test(str);
}
