import {
    getATAApi,
    getAirLinesListApi,
    getAircraftNumbersApi,
    getAircraftTerminalsApi,
    getBasicParamsListApi,
    getEngineListApi,
    getFlightNumApi,
    getPlaneModelsRepairApi,
    getRcOptionsApi,
    getRepairUnitApi
} from 'mocp/api/base'
import { getSysSeatListApi } from 'mocp/api/system'
import { uniqueArray } from 'mocp/utils/tool'
import { defineStore } from 'pinia'

const useBaseStore = defineStore('base', {
    state: () => {
        return {
            rcOptions: {
                assurePhase: [],
                commentConclusion: [],
                delayNature: [],
                delayProperty: [],
                eventType: [],
                flightImpact: [],
                liabilityNature: [],
                majorClass: [],
                occurArea: [],
                runPhase: [],
                stopReason: []
            },
            selectList: {
                deviceNumList: [], //机号
                terminalList: [], //航站下拉数据
                airlineCodeList: [], //航司下拉数据
                flightNumList: [], //航班号下拉数据
                typeList: [], //机型下拉数据
                ATAList: [], //章节下拉数据
                engineList: [], //发动机型号列表
                repairUnitList: [], //责任单位列表
                allSeatSelect: [] //席位
            },
            allBasicParams: {} //系统基础参数
        }
    },
    getters: {
        //格式化航站数据
        getTerminalObject(state) {
            return state.selectList.terminalList?.reduce((q, w) => {
                if (!q[w.terminalCode]) {
                    q[w.terminalCode] = []
                }
                q[w.terminalCode] = w.terminalCity + '-' + w.terminalName + '-' + w.terminalCode
                return q
            }, {})
        },
        getTerminalSelect(state) {
            return state.selectList.terminalList?.map((z) => ({
                value: z.terminalCode,
                label: z.terminalCity + '-' + z.terminalName + '-' + z.terminalCode
            }))
        },
        //格式化航司数据
        getAirlineSelect(state) {
            return state.selectList.airlineCodeList.map((z) => ({
                value: z.zOP3,
                zOP2: z.value,
                label: z.zOP3 + z.shortName
            }))
        },
        //格式化机型数据
        getTypeSelect(state) {
            return state.selectList.typeList.map((a) => ({
                label: a,
                value: a
            }))
        },
        //格式化章节下拉数据
        getAllATASelect(state) {
            return state.selectList.ATAList.map((q) => ({
                label: q.label + `(${q.zstortGc})`,
                value: q.label,
                id: q.id
            }))
        },
        getATASelect(state) {
            return uniqueArray(
                state.selectList.ATAList.map((q) => ({
                    label: q.label,
                    value: q.label
                }))
            )
        },
        getATAChecked: (state) => {
            return (model) => {
                let selectATAList = []
                if (model) {
                    selectATAList = state.selectList.ATAList.filter((item) => item.zstortGc == model)
                } else {
                    selectATAList = state.selectList.ATAList
                }
                return uniqueArray(
                    selectATAList.map((q) => ({
                        id: q.id,
                        label: q.label,
                        value: q.label
                    }))
                )
            }
        },
        //格式化发动机型号数据
        getEngineSelect(state) {
            return state.selectList.engineList.map((v) => {
                return {
                    label: v.zMETYPE,
                    value: v.zMETYPE
                }
            })
        },
        //获取席位
        getSeat(state) {
            return (seat) => {
                return (
                    seat
                        ?.split(',')
                        .map((item) => {
                            const seat = state.selectList.allSeatSelect.find((option) => item == option.id)
                            return seat?.seatName || ''
                        })
                        .join(',') || '-'
                )
            }
        }
    },
    actions: {
        async initSysData() {
            // uni.showLoading({
            //     title: '正在加载系统所有基本参数中，请稍后！',
            //     mask: true
            // })
            try {
                await Promise.all([
                    this.getAircraftNumbers(),
                    this.getBasicParamsList(),
                    this.getAircraftTerminals(),
                    this.getAirLinesList(),
                    this.getFlightNum(),
                    this.getPlaneModelsRepair(),
                    this.getATA(),
                    this.getEngineList(),
                    this.getRepairUnit(),
                    this.getSysSeatList(),
                    this.getRcOptions()
                ])
            } finally {
                uni.hideLoading()
            }
        },
        async getAircraftNumbers() {
            const res = await getAircraftNumbersApi({})
            if (res.code == 200) {
                this.selectList.deviceNumList = ['0000', ...res.data]
            }
        },
        async getAircraftTerminals() {
            const res = await getAircraftTerminalsApi({})
            if (res.code == 200) {
                this.selectList.terminalList = res.data
            }
        },
        async getAirLinesList() {
            const res = await getAirLinesListApi({})
            if (res.code == 200) {
                this.selectList.airlineCodeList = res.data
                    .filter((a) => a.sortId !== '0')
                    .sort((q, w) => Number(q.sortId) - Number(w.sortId))
                    .map((z) => ({
                        value: z.zOP2,
                        label: z.zOP2,
                        name: z.zCNAME20,
                        zOP3: z.zOP3,
                        shortName: z.reduceName,
                        zCFNAME: z.zCFNAME
                    }))
            }
        },
        async getFlightNum() {
            const res = await getFlightNumApi({})
            if (res.code == 200) {
                this.selectList.flightNumList = res.data
            }
        },
        async getBasicParamsList() {
            const res = await getBasicParamsListApi({})
            if (res.code == 200) {
                this.allBasicParams = res.data.reduce((q, w) => {
                    if (!q[w.paramType]) {
                        q[w.paramType] = []
                    }
                    q[w.paramType].push(w)
                    return q
                }, {})
            }
        },
        async getPlaneModelsRepair() {
            const res = await getPlaneModelsRepairApi({})
            if (res.code == 200) {
                this.selectList.typeList = res.data
            }
        },
        async getATA() {
            const res = await getATAApi({})
            if (res.code == 200) {
                this.selectList.ATAList = res.data.ataList
            }
        },
        async getEngineList() {
            const res = await getEngineListApi({})
            if (res.code == 200) {
                this.selectList.engineList = res.data
            }
        },
        async getRepairUnit() {
            const res = await getRepairUnitApi({})
            if (res.code == 200) {
                this.selectList.repairUnitList = res.data
            }
        },
        async getSysSeatList() {
            const res = await getSysSeatListApi({ pid: 0, pageSize: 9999 })
            if (res.code == 200) {
                this.selectList.allSeatSelect = res.data.list || []
            }
        },
        async getRcOptions() {
            const res = await getRcOptionsApi({})
            if (res.code == 200) {
                this.rcOptions.assurePhase = res.data.assurePhase || []
                this.rcOptions.commentConclusion = res.data.commentConclusion || []
                this.rcOptions.delayNature = res.data.delayNature || []
                this.rcOptions.delayProperty = res.data.delayProperty || []
                this.rcOptions.eventType = res.data.eventType || []
                this.rcOptions.flightImpact = res.data.flightImpact || []
                this.rcOptions.liabilityNature = res.data.liabilityNature || []
                this.rcOptions.majorClass = res.data.majorClass || []
                this.rcOptions.occurArea = res.data.occurArea || []
                this.rcOptions.runPhase = res.data.runPhase || []
                this.rcOptions.stopReason = res.data.stopReason || []
            }
        }
    },
    // 配置持久化
    persist: {
        // 调整为兼容多端的API
        storage: {
            setItem(key, value) {
                uni.setStorageSync(key, value)
            },
            getItem(key) {
                return uni.getStorageSync(key)
            }
        }
    }
})
export default useBaseStore
