// utils/tool.js
import Day from './dayjs'

/**
 * 时间戳格式化
 * 1715072168340 => 2024-05-07 16:56:08
 * format=YYYY/MM/DD HH:mm:ss
 */
export const timeStampFormat = (timeStamp, opt) => {
    if (!timeStamp || timeStamp == -1 || timeStamp == '-1' || timeStamp == 0 || timeStamp == '0') {
        return ''
    }
    timeStamp = parseInt(String(timeStamp))
    const format = opt?.format || 'YYYY/MM/DD HH:mm:ss'
    return Day(timeStamp).format(format)
}

/**
 *接受一个对象并返回一个只包含非空和非 null 值的新对象
 */
export const filterEmptyValues = (obj) => {
    return Object.entries(obj)
        .filter(([_, value]) => value !== null && value !== undefined && value !== '')
        .reduce((acc, [key, value]) => {
            acc[key] = value
            return acc
        }, {})
}

/**
 *数组去重
 */
export const uniqueArray = (arr) => {
    let uniqueArr = []
    let obj = {}
    for (let i = 0; i < arr.length; i++) {
        let str = JSON.stringify(arr[i])
        if (!obj[str]) {
            uniqueArr.push(arr[i])
            obj[str] = true
        }
    }
    return uniqueArr
}

/**
 * 文件预览
 * @param {*} fileUrl
 */
export const previewFile = (fileUrl) => {
    uni.showLoading({
        title: '正在打开文件...',
        mask: true
    })
    uni.downloadFile({
        url: fileUrl,
        success: function (res) {
            var filePath = res.tempFilePath
            uni.openDocument({
                filePath: filePath,
                showMenu: true,
                success: function () {
                    uni.hideLoading()
                    console.log('打开文件成功')
                },
                fail: function () {
                    uni.hideLoading()
                    uni.$mocpMessage.error('打开文件失败,请重新打开')
                }
            })
        },
        fail: function () {
            uni.hideLoading()
            uni.$mocpMessage.error('打开文件失败,请重新打开')
        }
    })
}

/**
 * 文件下载
 * @param {*} fileUrl
 */
export const downloadFile = (fileUrl) => {
    return new Promise((resolve, reject) => {
        uni.showLoading({
            mask: true,
            title: '保存中...'
        })
        uni.downloadFile({
            url: fileUrl,
            success: (res) => {
                if (res.statusCode === 200) {
                    uni.saveImageToPhotosAlbum({
                        filePath: res.tempFilePath,
                        success: () => {
                            uni.$mocpMessage.success('文件下载成功，已保存到相册')
                            resolve(res.tempFilePath)
                        },
                        fail: (err) => {
                            uni.$mocpMessage.error('保存到相册失败:', err)
                            reject(err)
                        },
                        complete: () => {
                            uni.hideLoading()
                        }
                    })
                } else {
                    uni.$mocpMessage.error('文件下载失败')
                    reject(new Error('文件下载失败'))
                }
            },
            fail: (err) => {
                uni.hideLoading()
                uni.$mocpMessage.error('下载文件失败:', err)
                reject(err)
            }
        })
    })
}

/**
 *  如果至少有一个参数被填写，那么所有参数都必须被填写
 * @param  {...any} params
 * @returns
 */
export const validateParameters = (...params) => {
    // 检查参数数组中是否至少有一个非空（非undefined、非null、非空字符串）的值
    const anyFilled = params.some((param) => param !== undefined && param !== null && param !== '')
    // 如果至少有一个参数被填写，那么检查是否所有的参数都被填写
    if (anyFilled) {
        // 使用every方法检查所有参数是否都被填写
        const allFilled = params.every((param) => param !== undefined && param !== null && param !== '')
        return allFilled
    }
    // 如果没有任何参数被填写，那么也是有效的
    return true
}
