// 引入 TreeRenderResult 类型，该类型定义在 @/lib/XMLProcessor/src/typing 模块中
import { TreeReconstructed, TreeRenderResult, TreeRenderResultFlatted } from '@/lib/XMLProcessor/src/typing'
import { md5 } from 'js-md5'
import { getUUID } from '../utils/uuid.ts'
// @ts-ignore
import { diffLines } from 'diff'
import {reconstructTree} from '@/lib/XMLProcessor/src/core/buildTree.ts'

// 定义 Processing 类，用于处理 XML 数据
export class Processing {
    // 私有属性，用于解析 XML 字符串为 DOM 对象
    private domParser: DOMParser
    private xmlSerializer: XMLSerializer

    // 构造函数，初始化 DOMParser 实例
    constructor() {
        this.domParser = new DOMParser()
        this.xmlSerializer = new XMLSerializer()
    }

    serializeXML(xmlDOM: Document): string {
        return this.xmlSerializer.serializeToString(xmlDOM)
    }

    private flattenTree(renderResult: TreeRenderResult[]): TreeRenderResultFlatted[] {
        const resp: TreeRenderResultFlatted[] = []
        for (const item of renderResult) {
            resp.push({
                key: item.key,
                label: item.label,
                hash: item.hash,
                index: item.index,
                chained: item.chained,
                textContent: item.textContent,
                node: item.node
            })
            if (item.children) {
                resp.push(...this.flattenTree(item.children))
            }
        }
        return resp
    }

    dualCompareFromString(xmlStringOld: string, xmlStringNew: string, bothHandledNode: string[], contentHoldNode: string[]) {
        const treeA = this.domParser.parseFromString(xmlStringOld, 'text/xml')
        const treeB = this.domParser.parseFromString(xmlStringNew, 'text/xml')
        const resultA = this.innerHandle(treeA.documentElement, bothHandledNode, contentHoldNode)
        const resultB = this.innerHandle(treeB.documentElement, bothHandledNode, contentHoldNode)
        treeA.documentElement.querySelectorAll(':not([data-key])').forEach((node) => node.remove())
        treeB.documentElement.querySelectorAll(':not([data-key])').forEach((node) => node.remove())
        // 扁平化所有节点
        const resultAFlatted = this.flattenTree(resultA)
        const resultBFlatted = this.flattenTree(resultB)

        const resultATree = resultAFlatted.map((item) => `${item.hash}`)
        const resultBTree = resultBFlatted.map((item) => `${item.hash}`)

        const diffResp = diffLines(resultATree.join('\n'), resultBTree.join('\n'), {
            ignoreWhitespace: true,
            ignoreNewlineAtEof: true,
            stripTrailingCr: true,
            newlineIsToken: false
        }).map((item: any) => ({
            ...item,
            value: item.value.split('\n').filter((a: string) => a.length > 0)
        }))

        let constructNodeA: TreeReconstructed[] = []
        let constructNodeB: TreeReconstructed[] = []

        for (let i = 0; i < diffResp.length; i++) {
            const currentItem = diffResp[i]
            if (!currentItem.added && !currentItem.removed) {
                // 由于完全相同，构建相同节点即可
                let treeStartForA = resultAFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeA.find((v) => v.key === a.key))
                if (treeStartForA !== -1) {
                    let aMapped = currentItem.value
                        .map((_: any, index: number) => {
                            return resultAFlatted[treeStartForA + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: null }))
                    constructNodeA.push(...aMapped)
                }
                let treeStartForB = resultBFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeB.find((v) => v.key === a.key))
                if (treeStartForB !== -1) {
                    let bMapped = currentItem.value
                        .map((_: any, index: number) => {
                            return resultBFlatted[treeStartForB + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: null }))
                    constructNodeB.push(...bMapped)
                }
                continue
            }
            const nextTerm = diffResp[i + 1]
            if (!nextTerm) {
                if (currentItem.added) {
                    let treeStartA = resultBFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeB.find((v) => v.key === a.key))
                    if (treeStartA !== -1) {
                        let aMapped = currentItem.value
                            .map((_: any, index: number) => {
                                return resultBFlatted[treeStartA + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: 'added' }))
                        constructNodeB.push(...aMapped)
                        constructNodeA.push(...aMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                } else if (currentItem.removed) {
                    let treeStartB = resultAFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeA.find((v) => v.key === a.key))
                    if (treeStartB !== -1) {
                        let bMapped = currentItem.value
                            .map((_: any, index: number) => {
                                return resultAFlatted[treeStartB + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: 'removed' }))
                        constructNodeA.push(...bMapped)
                        constructNodeB.push(...bMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                }
                continue
            }
            // 出现修改，则应该出现added和removed成对出现的情况
            if (currentItem.added && nextTerm.removed) {
                const minLength = Math.min(currentItem.value.length, nextTerm.value.length)
                let treeStartA = resultBFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeB.find((v) => v.key === a.key))
                if (treeStartA !== -1) {
                    let aMapped = currentItem.value
                        .slice(0, minLength)
                        .map((_: any, index: number) => {
                            return resultBFlatted[treeStartA + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'deleted' }))
                    constructNodeB.push(...aMapped)
                }
                let treeStartB = resultAFlatted.findIndex((a) => a.hash === nextTerm.value[0] && !constructNodeA.find((v) => v.key === a.key))
                if (treeStartB !== -1) {
                    let bMapped = nextTerm.value
                        .slice(0, minLength)
                        .map((_: any, index: number) => {
                            return resultAFlatted[treeStartB + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'added' }))
                    constructNodeA.push(...bMapped)
                }
                // 根据情况插入对齐
                if (currentItem.value.length > nextTerm.value.length) {
                    const allExtraAdded = currentItem.value.slice(minLength)
                    let treeAddedForA = resultBFlatted.findIndex((a) => a.hash === allExtraAdded[0] && !constructNodeB.find((v) => v.key === a.key))
                    if (treeAddedForA !== -1) {
                        let aMapped = allExtraAdded
                            .map((_: any, index: number) => {
                                return resultBFlatted[treeAddedForA + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: null }))
                        constructNodeB.push(...aMapped)
                        // 再为A补齐
                        constructNodeA.push(...aMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                } else if (currentItem.value.length < nextTerm.value.length) {
                    const allExtraRemoved = nextTerm.value.slice(minLength)
                    let treeAddedForB = resultAFlatted.findIndex((a) => a.hash === allExtraRemoved[0] && !constructNodeA.find((v) => v.key === a.key))
                    if (treeAddedForB !== -1) {
                        let bMapped = allExtraRemoved
                            .map((_: any, index: number) => {
                                return resultAFlatted[treeAddedForB + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: 'deleted' }))
                        constructNodeA.push(...bMapped)
                        constructNodeB.push(...bMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                }
                i++
            } else if (currentItem.removed && nextTerm.added) {
                const minLength = Math.min(currentItem.value.length, nextTerm.value.length)
                let treeStartA = resultAFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeA.find((v) => v.key === a.key))
                if (treeStartA !== -1) {
                    let aMapped = currentItem.value
                        .slice(0, minLength)
                        .map((_: any, index: number) => {
                            return resultAFlatted[treeStartA + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'removed' }))
                    constructNodeA.push(...aMapped)
                }
                let treeStartB = resultBFlatted.findIndex((a) => a.hash === nextTerm.value[0] && !constructNodeB.find((v) => v.key === a.key))
                if (treeStartB !== -1) {
                    let bMapped = nextTerm.value
                        .slice(0, minLength)
                        .map((_: any, index: number) => {
                            return resultBFlatted[treeStartB + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'added' }))
                    constructNodeB.push(...bMapped)
                }
                if (currentItem.value.length > nextTerm.value.length) {
                    const allExtraRemoved = currentItem.value.slice(minLength)
                    let treeAddedForB = resultAFlatted.findIndex((a) => a.hash === allExtraRemoved[0] && !constructNodeA.find((v) => v.key === a.key))
                    if (treeAddedForB !== -1) {
                        let bMapped = allExtraRemoved
                            .map((_: any, index: number) => {
                                return resultAFlatted[treeAddedForB + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: 'removed' }))
                        constructNodeA.push(...bMapped)
                        constructNodeB.push(...bMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                } else if (currentItem.value.length < nextTerm.value.length) {
                    const allExtraAdded = nextTerm.value.slice(minLength)
                    let treeAddedForA = resultBFlatted.findIndex((a) => a.hash === allExtraAdded[0] && !constructNodeB.find((v) => v.key === a.key))
                    if (treeAddedForA !== -1) {
                        let aMapped = allExtraAdded
                            .map((_: any, index: number) => {
                                return resultBFlatted[treeAddedForA + index]
                            })
                            .map((a: any) => Object.assign({}, a, { type: 'added' }))
                        constructNodeB.push(...aMapped)
                        constructNodeA.push(...aMapped.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                    }
                }
                i++
            }
            // 如果不是出现成对的情况，则需要直接补位
            // 额外添加的，旧的树需要添加
            else if (currentItem.added && !nextTerm.removed) {
                const addedForNew = resultBFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeB.find((v) => v.key === a.key))
                if (addedForNew > -1) {
                    let mappedData = currentItem.value
                        .map((_: any, index: number) => {
                            return resultBFlatted[addedForNew + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'added' }))
                    constructNodeB.push(...mappedData)
                    constructNodeA.push(...mappedData.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                }
            } else if (currentItem.removed && !nextTerm.added) {
                const removedForOld = resultAFlatted.findIndex((a) => a.hash === currentItem.value[0] && !constructNodeA.find((v) => v.key === a.key))
                if (removedForOld > -1) {
                    let mappedData = currentItem.value
                        .map((_: any, index: number) => {
                            return resultAFlatted[removedForOld + index]
                        })
                        .map((a: any) => Object.assign({}, a, { type: 'removed' }))
                    constructNodeA.push(...mappedData)
                    constructNodeB.push(...mappedData.map((a: any) => Object.assign({}, a, { type: 'placeholder' })))
                }
            }
        }
        const treeOld = reconstructTree(constructNodeA, contentHoldNode)
        const treeNew = reconstructTree(constructNodeB, contentHoldNode)
        console.log('the resp', {constructNodeA, constructNodeB})
        return {
            treeOld, treeNew
        }
    }

    // 处理 XML 字符串的方法，返回树形数据和 XML DOM 对象
    processXML(
        xmlContent: string,
        handledNode: string[],
        contentHoldNode: string[]
    ): {
        treeData: TreeRenderResult[]
        xmlDOM: Document
        xmlContent: string
    } {
        // 使用 DOMParser 解析 XML 字符串
        const parsedTree = this.domParser.parseFromString(xmlContent, 'text/xml')
        // 获取 XML 文档的根元素
        const rootElement = parsedTree.documentElement
        const result = {
            // 调用 innerHandle 方法处理根元素，生成树形数据
            treeData: this.innerHandle(rootElement, handledNode, contentHoldNode),
            // 返回解析后的 XML DOM 对象
            xmlDOM: parsedTree
        }
        parsedTree.documentElement.querySelectorAll(':not([data-key])').forEach((node) => node.remove())
        return {
            ...result,
            xmlContent: this.serializeXML(parsedTree)
        }
    }

    // 异步处理 XML 文件的方法，返回包含树形数据和 XML DOM 对象的 Promise
    async processFile(
        xmlFile: File,
        handledNode: string[],
        contentHoldNode: string[]
    ): Promise<{
        treeData: TreeRenderResult[]
        xmlDOM: Document
        xmlContent: string
    }> {
        // 检查文件是否为有效的 XML 文件
        if (!(xmlFile instanceof File) || !xmlFile.type.includes('xml')) {
            return Promise.reject('The file is not a valid XML file.')
        }
        return new Promise((resolve, reject) => {
            const fr = new FileReader()
            // 当文件读取完成时触发的事件处理函数
            fr.onload = () => {
                const content = fr.result as string
                try {
                    // 调用 processXML 方法处理读取的文件内容
                    const { treeData, xmlDOM, xmlContent } = this.processXML(content, handledNode, contentHoldNode)

                    // 解析成功，返回树形数据和 XML DOM 对象
                    resolve({ treeData, xmlDOM, xmlContent })
                } catch (error) {
                    // 解析失败，拒绝 Promise 并返回错误信息
                    reject(error)
                }
            }
            // 当文件读取出错时触发的事件处理函数
            fr.onerror = () => {
                reject('Error reading file.')
            }
            // 以文本形式读取文件
            fr.readAsText(xmlFile)
        })
    }

    // 私有方法，用于递归处理单个 DOM 节点，生成树形数据
    private innerHandle(domNode: Element, handledNode: string[], contentHoldNode: string[]): TreeRenderResult[] {
        let treeData: TreeRenderResult[] = []
        // 生成唯一的 key
        let targetKey = domNode.getAttribute('data-key')
        if (!targetKey) {
            targetKey = getUUID()
        }
        // 创建树形数据项
        const treeItem: TreeRenderResult = {
            key: targetKey,
            label: domNode.nodeName,
            children: [],
            index: 0,
            chained: [0],
            hash: contentHoldNode.includes(domNode.nodeName) ? md5.hex(domNode.textContent! || '') : md5(domNode.nodeName),
            textContent: domNode.textContent || '',
            node: domNode
        }
        // 为 DOM 节点设置 data-key 属性
        domNode.setAttribute('data-key', targetKey)
        domNode.setAttribute('data-w-e-type', domNode.nodeName)
        domNode.setAttribute('data-indent-level', '0')
        domNode.setAttribute('data-chained', treeItem.chained.join('$'))
        domNode.setAttribute('data-hash', treeItem.hash)
        // 递归处理子节点
        treeItem.children = this.innerGroupHandle(domNode.children, handledNode, [0], contentHoldNode)
        // 将树形数据项添加到结果数组中
        treeData.push(treeItem)

        return treeData
    }

    // 私有方法，用于递归处理一组 DOM 节点，生成树形数据
    private innerGroupHandle(nodeSet: HTMLCollection, handledNode: string[], startChained: number[], contentHoldNode: string[]): TreeRenderResult[] {
        let targetResult: TreeRenderResult[] = []
        let realIndex = -1
        // 遍历节点集合
        for (let i = 0; i < nodeSet.length; i++) {
            const node = nodeSet[i] as Element
            // 检查节点是否在处理列表中
            if (!handledNode.includes(node.nodeName)) {
                // 如果节点有子节点，递归处理子节点
                if (node.children.length) {
                    targetResult.push(...this.innerGroupHandle(node.children, handledNode, startChained, contentHoldNode))
                }
                continue
            }
            realIndex++
            // 生成唯一的 key
            let targetKey = node.getAttribute('data-key')
            if (!targetKey) {
                targetKey = getUUID()
            }
            // 创建树形数据项
            const treeItem: TreeRenderResult = {
                key: targetKey,
                label: node.nodeName,
                index: realIndex,
                chained: [...startChained, realIndex],
                hash: contentHoldNode.includes(node.nodeName) ? md5.hex(node.textContent! || '') : md5(node.nodeName),
                textContent: node.textContent || '',
                node: node
            }
            // 为 DOM 节点设置 data-key 属性
            node.setAttribute('data-key', targetKey)
            node.setAttribute('data-w-e-type', node.nodeName)
            node.setAttribute('data-indent-level', startChained.length.toString())
            node.setAttribute('data-chained', treeItem.chained.join('$'))
            node.setAttribute('data-hash', treeItem.hash)
            // 如果节点有子节点，递归处理子节点
            if (node.children.length > 0) {
                treeItem.children = this.innerGroupHandle(node.children, handledNode, [...startChained, realIndex], contentHoldNode)
            }
            // 将树形数据项添加到结果数组中
            targetResult.push(treeItem)
        }
        return targetResult
    }
}
