// 引入 TreeRenderResult 类型，该类型定义在 @/lib/XMLProcessor/src/typing 模块中
import { TreeRenderResult } from '@/lib/XMLProcessor/src/typing';
// 引入 UUID 类用于生成唯一标识符
import { UUID } from 'uuidjs';

// 定义 Processing 类，用于处理 XML 数据
export class Processing {
    // 私有属性，用于解析 XML 字符串为 DOM 对象
    private domParser: DOMParser;
    private xmlSerializer: XMLSerializer

    // 构造函数，初始化 DOMParser 实例
    constructor() {
        this.domParser = new DOMParser();
        this.xmlSerializer = new XMLSerializer();
    }

    serializeXML(xmlDOM: Document): string {
        return this.xmlSerializer.serializeToString(xmlDOM);
    }

    dualCompareFromString(xmlStringA: string, xmlStringB: string, bothHandledNode: string[]) {
        
    }

    // 处理 XML 字符串的方法，返回树形数据和 XML DOM 对象
    processXML(xmlContent: string, handledNode: string[]): {
        treeData: TreeRenderResult[];
        xmlDOM: Document;
        xmlContent: string
    } {
        // 使用 DOMParser 解析 XML 字符串
        const parsedTree = this.domParser.parseFromString(xmlContent, 'text/xml');
        // 获取 XML 文档的根元素
        const rootElement = parsedTree.documentElement;
        const result = {
            // 调用 innerHandle 方法处理根元素，生成树形数据
            treeData: this.innerHandle(rootElement, handledNode),
            // 返回解析后的 XML DOM 对象
            xmlDOM: parsedTree
        }
        parsedTree.documentElement.querySelectorAll(':not([data-key])').forEach((node) => node.remove())
        return {
            ...result,
            xmlContent: this.serializeXML(parsedTree)
        }
    }

    // 异步处理 XML 文件的方法，返回包含树形数据和 XML DOM 对象的 Promise
    async processFile(xmlFile: File, handledNode: string[]): Promise<{ treeData: TreeRenderResult[]; xmlDOM: Document;xmlContent: string }> {
        // 检查文件是否为有效的 XML 文件
        if (!(xmlFile instanceof File) || !xmlFile.type.includes('xml')) {
            return Promise.reject('The file is not a valid XML file.');
        }
        return new Promise((resolve, reject) => {
            const fr = new FileReader();
            // 当文件读取完成时触发的事件处理函数
            fr.onload = () => {
                const content = fr.result as string;
                try {
                    // 调用 processXML 方法处理读取的文件内容
                    const { treeData, xmlDOM, xmlContent } = this.processXML(content, handledNode);

                    // 解析成功，返回树形数据和 XML DOM 对象
                    resolve({ treeData, xmlDOM, xmlContent });
                } catch (error) {
                    // 解析失败，拒绝 Promise 并返回错误信息
                    reject(error);
                }
            };
            // 当文件读取出错时触发的事件处理函数
            fr.onerror = () => {
                reject('Error reading file.');
            };
            // 以文本形式读取文件
            fr.readAsText(xmlFile);
        });
    }

    // 私有方法，用于递归处理单个 DOM 节点，生成树形数据
    private innerHandle(domNode: Element, handledNode: string[]): TreeRenderResult[] {
        let treeData: TreeRenderResult[] = [];
        // 生成唯一的 key
        let targetKey = domNode.getAttribute('data-key');
        if (!targetKey) {
            targetKey = 'g-' + UUID.generate()
        }
        // 创建树形数据项
        const treeItem: TreeRenderResult = {
            key: targetKey,
            label: domNode.nodeName,
            children: [],
            index: 0,
            chained: [0]
        };
        // 为 DOM 节点设置 data-key 属性
        domNode.setAttribute('data-key', targetKey);
        domNode.setAttribute('data-w-e-type', domNode.nodeName)
        domNode.setAttribute('data-indent-level', '0')
        // 递归处理子节点
        treeItem.children = this.innerGroupHandle(domNode.children, handledNode, [0]);
        // 将树形数据项添加到结果数组中
        treeData.push(treeItem);

        return treeData;
    }

    // 私有方法，用于递归处理一组 DOM 节点，生成树形数据
    private innerGroupHandle(nodeSet: HTMLCollection, handledNode: string[], startChained: number[]): TreeRenderResult[] {
        let targetResult: TreeRenderResult[] = [];
        let realIndex = -1;
        // 遍历节点集合
        for (let i = 0; i < nodeSet.length; i++) {
            const node = nodeSet[i] as Element;
            // 检查节点是否在处理列表中
            if (!handledNode.includes(node.nodeName)) {
                // 如果节点有子节点，递归处理子节点
                if (node.children.length) {
                    targetResult.push(...this.innerGroupHandle(node.children, handledNode, startChained));
                }
                continue;
            }
            realIndex++;
            // 生成唯一的 key
            let targetKey = node.getAttribute('data-key');
            if (!targetKey) {
                targetKey = 'g-' + UUID.generate()
            }
            // 创建树形数据项
            const treeItem: TreeRenderResult = {
                key: targetKey,
                label: node.nodeName,
                index: realIndex,
                chained: [...startChained, realIndex]
            };
            // 为 DOM 节点设置 data-key 属性
            node.setAttribute('data-key', targetKey);
            node.setAttribute('data-w-e-type', node.nodeName)
            node.setAttribute('data-indent-level', startChained.length.toString())
            // 如果节点有子节点，递归处理子节点
            if (node.children.length > 0) {
                treeItem.children = this.innerGroupHandle(node.children, handledNode, [...startChained, realIndex]);
            }
            // 将树形数据项添加到结果数组中
            targetResult.push(treeItem);
        }
        return targetResult;
    }
}    