import {
    ContinuousRecord,
    GenericRecordType,
    IndexRecord,
    OperationRecordAlias,
    OperationRecordInternalState,
    OperationRecordTyping,
    RawRecordItem,
    RecordType,
    VectorRecord
} from "../types/OperationRecordTyping";
import {deepClone} from "../utils/object";

export class OperationRecord {

    private readonly operationRecordTypeSet: OperationRecordTyping
    private readonly operationRecordAlias: OperationRecordAlias
    private rawRecord: Array<RawRecordItem>
    private operationRecord: string
    private relativeWidth: number = 0
    private relativeHeight: number = 0
    private recordListener: Function

    constructor(recordListener: Function) {
        this.operationRecordTypeSet = {}
        this.operationRecordAlias = {}
        this.rawRecord = []
        this.operationRecord = ''
        this.recordListener = recordListener
    }

    registerType(name: string, type: RecordType, alias: string) {
        Reflect.set(this.operationRecordTypeSet, name, type)
        Reflect.set(this.operationRecordAlias, name, alias)
    }

    setRelativeGEO(width: number, height: number) {
        this.relativeWidth = width
        this.relativeHeight = height
    }

    public setNewRawRecord(records: Array<RawRecordItem>) {
        this.rawRecord = deepClone(records)
        this.operationRecord = ''
        this.rawRecord.forEach(recordItem => {
            this.storeRecord(recordItem.name, recordItem.record, false)
        })
    }

    storeRecord(name: string, record: GenericRecordType, recording = true) {
        const currentRecordType: RecordType = Reflect.get(this.operationRecordTypeSet, name)
        const currentRecordAlias: string = Reflect.get(this.operationRecordAlias, name)
        if (recording && ![RecordType.Revoke, RecordType.Recover].includes(currentRecordType)) {
            this.rawRecord.push({
                name,
                record
            })
        }
        this.operationRecord += this.makeRecordString(record, currentRecordAlias, currentRecordType)
        if (currentRecordType === RecordType.Recover || currentRecordType === RecordType.Revoke) {
            // @ts-ignore
            const slicedRecords = this.rawRecord.slice(0, this.rawRecord.length - record)
                .reduce((q, a)=>{
                    const recordType: RecordType = Reflect.get(this.operationRecordTypeSet, a.name)
                    const alias: string = Reflect.get(this.operationRecordAlias, a.name)
                    const record = this.makeRecordString(a.record, alias, recordType)
                    return q + record
                }, '')
            this.recordListener.call(null, slicedRecords)
        }
        else {
            this.recordListener.call(null, this.operationRecord)
        }
    }

    private makeRecordString(record: GenericRecordType, currentRecordAlias: string, currentRecordType: RecordType) {
        switch (currentRecordType) {
            case RecordType.Vector:
                const vectorRecord = (record as VectorRecord)
                return `${currentRecordAlias} ${(vectorRecord.sx / this.relativeWidth).toFixed(2)} ${(vectorRecord.sy / this.relativeHeight).toFixed(2)} ${(vectorRecord.ex / this.relativeWidth).toFixed(2)} ${(vectorRecord.ey / this.relativeHeight).toFixed(2)} `
            case RecordType.Continuous:
                const continuousRecord = (record as ContinuousRecord)
                return `${currentRecordAlias} ${continuousRecord.map(item => [(item.ex / this.relativeWidth).toFixed(2), (item.ey / this.relativeHeight).toFixed(2)]).flat().join(' ')} `
            case RecordType.Index:
                const indexRecord = (record as IndexRecord)
                return `${currentRecordAlias} ${indexRecord} `
            case RecordType.Immediate:
                return `${currentRecordAlias} `
            case RecordType.VectorPayload:
                const pointPayloadRecord = (record as VectorRecord)
                return `${currentRecordAlias} ${(pointPayloadRecord.sx / this.relativeWidth).toFixed(2)} ${(pointPayloadRecord.sy / this.relativeHeight).toFixed(2)} ${(pointPayloadRecord.ex / this.relativeWidth).toFixed(2)} ${(pointPayloadRecord.ey / this.relativeHeight).toFixed(2)} ${pointPayloadRecord.payload} `
            case RecordType.Recover:
            case RecordType.Revoke:
            case RecordType.UNSET:
            default:
                return ''
        }
    }

    parseRecord(inputRecord: string): Array<RawRecordItem> {
        let wholeRecord: Array<RawRecordItem> = []
        let inputRecordSplit: Array<string> = inputRecord.trim().split(' ')
        let paintCommandAlias: string
        let paintCommand: string
        try {
            inputRecordSplit.forEach(span => {
                if (!Object.values(this.operationRecordAlias).includes(span)) {
                    const lastInsertedRecord = wholeRecord[wholeRecord.length - 1]
                    const lastRecordType = this.operationRecordTypeSet[lastInsertedRecord.name]
                    if (lastRecordType === RecordType.Vector || lastRecordType === RecordType.VectorPayload) {
                        const objectKeyLength = Object.keys(lastInsertedRecord.record).length
                        if (objectKeyLength === 0) {
                            lastInsertedRecord.record.sx = Number(span) * this.relativeWidth
                            return
                        }
                        if (objectKeyLength === 1) {
                            lastInsertedRecord.record.sy = Number(span) * this.relativeHeight
                            return
                        }
                        if (objectKeyLength === 2) {
                            lastInsertedRecord.record.ex = Number(span) * this.relativeWidth
                            return
                        }
                        if (objectKeyLength === 3) {
                            lastInsertedRecord.record.ey = Number(span) * this.relativeHeight
                        }
                        if (lastRecordType === RecordType.VectorPayload) {
                            lastInsertedRecord.record.payload = span
                        } else {
                            lastInsertedRecord.record.payload = null
                        }
                        return;
                    }
                    if (lastRecordType === RecordType.Continuous) {
                        if (!Array.isArray(lastInsertedRecord.record)) {
                            lastInsertedRecord.record = []
                        }
                        const lastRecordExEyItem = lastInsertedRecord.record[lastInsertedRecord.record.length - 1]
                        if (!lastRecordExEyItem || lastRecordExEyItem.ey) {
                            const newlyAddedItem = {
                                ex: Number(span) * this.relativeWidth
                            }
                            lastInsertedRecord.record.push(newlyAddedItem)
                            return
                        }
                        if (!lastRecordExEyItem.ey) {
                            lastRecordExEyItem.ey = Number(span) * this.relativeHeight
                        }
                        return
                    }
                    if (lastRecordType === RecordType.Index) {
                        lastInsertedRecord.record = Number(span)
                        return
                    }
                    lastInsertedRecord.record = null
                    return
                }

                paintCommandAlias = span[0]
                paintCommand = Object.keys(this.operationRecordAlias).find(key => this.operationRecordAlias[key] === paintCommandAlias) as string
                let currentFoundRecord: RawRecordItem = {
                    name: '',
                    record: {}
                }
                currentFoundRecord.name = paintCommand
                wholeRecord.push(currentFoundRecord)
            })
        } catch (e) {
            console.log('error???', e)
            return []
        }
        return wholeRecord
    }

    removeRecord() {
        this.rawRecord.length = 0
        this.operationRecord = ''
    }

    public getInternalState(): OperationRecordInternalState {
        return {
            operationRecordTypeSet: this.operationRecordTypeSet,
            operationRecord: this.operationRecord,
            operationRecordAlias: this.operationRecordAlias,
            rawRecord: this.rawRecord
        }
    }
}