import { createAlova } from 'alova'
import vueHook from 'alova/vue'
import GlobalFetch from 'alova/GlobalFetch'
import { API_URL, ApiConfig } from '../../project.app.config'
import qs from 'qs'
import { createAlovaMockAdapter } from '@alova/mock'
import mock from './mock'
import useUserStore from '@/store/user'
import { Message } from '@arco-design/web-vue'
import useAppStore from '@/store/app/'

// 创建模拟请求适配器
const mockAdapter = createAlovaMockAdapter([mock], {
    enable: true,
    delay: ApiConfig.MOCK_DELAY,
    httpAdapter: GlobalFetch(), // 无适配使用接口请求
    mockRequestLogger: true // 是否打印mock接口请求信息
})

// 创建实例
export const alova = createAlova({
    baseURL: API_URL,
    statesHook: vueHook,
    shareRequest: true,

    // 通过环境变量控制生产环境下，不会将mock相关代码打包进去
    requestAdapter: process.env.NODE_ENV === 'development' ? mockAdapter : GlobalFetch(),

    // 请求拦截器
    beforeRequest(method) {
        //开启加载动画
        const appStore = useAppStore()
        const loading = method.meta?.loading
        if (loading || loading === '') {
            if (typeof loading === 'string' && loading !== '') {
                appStore.loadText = loading
            }
            appStore.count++
        }
        // 设置请求头application/json;charset=UTF-8
        const isUpload = method.config?.meta?.isUpload
        console.log('the upload here', isUpload, method)
        if (isUpload && method.data) {
            method.data = Object.entries(method.data).reduce(function (q, w) {
                if (!(w[1] instanceof File)) {
                    q.set(w[0], w[1])
                } else {
                    q.set(w[0], w[1], w[1].name)
                }
                return q
            }, new FormData())
        } else {
            method.config.headers = {
                'Content-Type': 'application/x-www-form-urlencoded'
            }
            //转换成表单数据
            method.data = qs.stringify(method.data)
        }
        //加入token
        const userStore = useUserStore()
        if (userStore.token) {
            method.config.headers.Authorization = `Bearer ${userStore.token}`
        }
    },

    // 响应拦截器
    responded: {
        // 当使用GlobalFetch请求适配器时，第一个参数接收Response对象
        onSuccess: async (response, method) => {
            if (response.status >= 400) {
                throw new Error(response.statusText)
            }
            if (method.meta.blob) {
                return await response.blob()
            }
            const json = await response.json()
            if ([402].includes(json.code)) {
                //接口异常
                Message.error(json.message)
                const userStore = useUserStore()
                userStore.handleLogOut()
                return Promise.reject(new Error(json.message))
            } else {
                return json
            }
        },
        // 请求错误时将会进入该拦截器
        onError(error) {
            console.log(error)
            throw error
        },
        // 当你需要在请求不论是成功、失败、还是命中缓存都需要执行的逻辑时，可以在创建alova实例时指定全局的`onComplete`拦截器，例如关闭请求 loading 状态。
        onComplete: async () => {
            // 处理响应完成后的逻辑
            // console.log("执行完成");
            //关闭加载动画
            const appStore = useAppStore()
            appStore.count--
            if (appStore.count <= 0) {
                appStore.count = 0
                appStore.loadText = '加载中...'
            }
        }
    }
})
