import {plantCanvas} from "./renderBasic";
import {GenericDrawFunc} from "../types/HandlerFunc";
import {clamp} from "../utils/math";
import {ProcessCycle, ProcessCycleItem, ProcessCycleMapping} from "../types/HandleProcessCycle";
import {isDrawFromAction, setDrawFromAction} from "../state/drawOperationType";
import {OperationRecord} from "./operationRecord";
import {enableRecover, rollbackStep} from "../state/canvasRecover";
import {CoreInternalState} from "../types/CoreOperationTyping";

export class CoreOperation {
    private readonly defaultNoop: string = 'noop'
    private currentDrawOption: string = 'noop'
    private currentImmediateOption: string | undefined = 'noop'
    private currentDrawAttachment: any = null
    private isMouseDown: boolean = false
    private element: HTMLElement
    private readonly cvs_width: number
    private readonly cvs_height: number
    private sx: number | undefined
    private sy: number | undefined
    private ex: number | undefined
    private ey: number | undefined
    public record: OperationRecord | undefined

    private readonly persistedStartPs: Array<ProcessCycleItem> = []
    private readonly persistedStampPs: Array<ProcessCycleItem> = []
    private readonly persistedStopPs: Array<ProcessCycleItem> = []
    private readonly immediatePs: Array<ProcessCycleItem> = []
    private readonly beforeStartPs: Array<ProcessCycleItem> = []
    private readonly startProcess: ProcessCycleMapping
    private readonly stampProcess: ProcessCycleMapping
    private readonly stopProcess: ProcessCycleMapping
    private isHandleImmediate: boolean

    constructor(canvas: HTMLElement) {
        this.element = canvas
        this.startProcess = {}
        this.stampProcess = {}
        this.stopProcess = {}
        this.isHandleImmediate = false
        this.record = undefined
        const innerCanvas = this.element.querySelector('canvas')
        // @ts-ignore
        this.cvs_width = innerCanvas.width
        // @ts-ignore
        this.cvs_height = innerCanvas.height
        this.addEventListener()
    }

    setRecord(record: OperationRecord) {
        this.record = record
    }

    triggerDraw(name: string) {
        const immediateFunc: ProcessCycleItem | undefined = this.immediatePs.find(item => item.name === name)
        this.isHandleImmediate = typeof immediateFunc !== 'undefined'
        // 立即触发类方法不会主动调用本方法，而会调用其他所有常驻方法，
        if (typeof immediateFunc !== 'undefined') {
            this.currentDrawAttachment = immediateFunc.attachment
            this.currentImmediateOption = immediateFunc.name
            this.immediatePs.filter(item => item.name !== immediateFunc.name && item.persist).forEach(f => {
                f.func.call(null)
            })
        } else {
            this.currentDrawOption = name
        }
        this.beforeStartPs.forEach(item => {
            item.func.call(null)
        })
    }

    register(cycle: ProcessCycle, name: string, func: GenericDrawFunc, attachment: Object, persist?: boolean) {
        switch (cycle) {
            case ProcessCycle.Immediate:
                this.immediatePs.push({func, attachment, name, persist})
                return;
            case ProcessCycle.BeforeStart:
                this.beforeStartPs.push({func})
                return;
            case ProcessCycle.Start:
                if (persist)
                    this.persistedStartPs.push({func, attachment})
                else
                    Reflect.set(this.startProcess, name, {func, attachment})
                return
            case ProcessCycle.Stamp:
                if (persist)
                    this.persistedStampPs.push({func, attachment})
                else
                    Reflect.set(this.stampProcess, name, {func, attachment})
                return;
            case ProcessCycle.Stop:
                if (persist)
                    this.persistedStopPs.push({func, attachment})
                else
                    Reflect.set(this.stopProcess, name, {func, attachment})
                return;
            default:
                break
        }
    }

    stopDraw() {
        this.currentDrawOption = this.defaultNoop
    }

    // 返回所有私有内部成员变量，不仅可以抽取数据，而且可以直接发起绘图
    public getInternalState(): CoreInternalState {
        return {
            sx: this.sx,
            sy: this.sy,
            ex: this.ex,
            ey: this.ey,
            isMouseDown: this.isMouseDown,
            currentDrawOption: this.currentDrawOption,
            currentDrawAttachment: this.currentDrawAttachment,
            currentImmediateOption: this.currentImmediateOption,
            startProcess: this.startProcess,
            stampProcess: this.stampProcess,
            stopProcess: this.stopProcess,
            persistedStartPs: this.persistedStartPs,
            persistedStampPs: this.persistedStampPs,
            persistedStopPs: this.persistedStopPs,
            immediatePs: this.immediatePs,
            isHandleImmediate: this.isHandleImmediate
        }
    }

    private endDraw() {
        if (!this.isMouseDown) return
        this.persistedStopPs.forEach(item => {
            if (item.func.length === 5)
                item.func.call(null, this.sx, this.sy, this.ex, this.ey, item.attachment)
            else
                item.func.call(null)
        })
        const currentOp: ProcessCycleItem = Reflect.get(this.stopProcess, this.currentDrawOption)
        if (typeof currentOp !== 'undefined') {
            currentOp.func.call(null, this.sx, this.sy, this.ex, this.ey, currentOp.attachment)
        }
        plantCanvas()
        this.sx = undefined
        this.sy = undefined
        this.ex = undefined
        this.ey = undefined
        this.isMouseDown = false
        this.currentDrawAttachment = null
    }

    private handleStart(sx: number, sy: number, rx?: number, ry?: number) {
        if (this.currentDrawOption === this.defaultNoop) return
        if (!isDrawFromAction()) {
            setDrawFromAction()
            plantCanvas()
            const toBeCut = this.record?.getInternalState().rawRecord
            // @ts-ignore
            this.record?.setNewRawRecord(toBeCut?.slice(0, toBeCut?.length - rollbackStep.value))
            rollbackStep.value = 0
            enableRecover.value = false
        }
        this.triggerDraw(this.currentDrawOption)
        this.isMouseDown = true
        this.sx = sx
        this.sy = sy
        const currentOp: ProcessCycleItem = Reflect.get(this.startProcess, this.currentDrawOption)
        if (typeof currentOp !== 'undefined') {
            this.currentDrawAttachment = currentOp.attachment
            if (Reflect.has(this.currentDrawAttachment, 'realPoint')) {
                currentOp.func.call(null, rx, ry, this.ex, this.ey, currentOp.attachment)
            } else {
                currentOp.func.call(null, this.sx, this.sy, this.ex, this.ey, currentOp.attachment)
            }
        }
        this.persistedStartPs.forEach(item => {
            item.func.call(null, this.sx, this.sy, this.ex, this.ey, item.attachment)
        })
    }

    private handleMove(ex: number, ey: number) {
        if (this.currentDrawOption === this.defaultNoop || !this.isMouseDown) return
        this.ex = ex
        this.ey = ey
        const currentOp: ProcessCycleItem = Reflect.get(this.stampProcess, this.currentDrawOption)
        if (typeof currentOp !== 'undefined')
            currentOp.func.call(null, this.sx, this.sy, this.ex, this.ey, currentOp.attachment)
        this.persistedStampPs.forEach(item => {
            item.func.call(null, this.sx, this.sy, this.ex, this.ey, item.attachment)
        })
    }

    private addEventListener() {

        const genericMobileHandle = (e: TouchEvent): { sx: number, sy: number, rx: number, ry: number } => {
            const {left, top} = this.element.getBoundingClientRect()
            const sx = clamp(0, e.touches[0].clientX - left, this.element.clientWidth)
            const sy = clamp(0, e.touches[0].clientY - top, this.element.clientHeight)
            const realX = sx / this.element.clientWidth * this.cvs_width
            const realY = sy / this.element.clientHeight * this.cvs_height
            return {
                sx: realX,
                sy: realY,
                rx: sx,
                ry: sy
            }
        }

        const genericDesktopHandle = (e: MouseEvent): { sx: number, sy: number, rx: number, ry: number } => {
            const sx = clamp(0, e.offsetX, this.element.clientWidth)
            const sy = clamp(0, e.offsetY, this.element.clientHeight)
            const realX = sx / this.element.clientWidth * this.cvs_width
            const realY = sy / this.element.clientHeight * this.cvs_height
            return {
                sx: realX,
                sy: realY,
                rx: sx,
                ry: sy
            }
        }

        document.body.addEventListener('touchmove', e => {
            e.preventDefault()
        })

        this.element.addEventListener('mousedown', (e: MouseEvent) => {
            const {sx: realX, sy: realY, rx, ry} = genericDesktopHandle(e)
            this.handleStart(realX, realY, rx, ry)
        })
        this.element.addEventListener('mousemove', (e: MouseEvent) => {
            const {sx: realX, sy: realY} = genericDesktopHandle(e)
            this.handleMove(realX, realY)
        })
        this.element.addEventListener('mouseup', this.endDraw.bind(this))
        this.element.addEventListener('mouseleave', this.endDraw.bind(this))

        this.element.addEventListener('touchstart', e => {
            const {sx: realX, sy: realY, rx, ry} = genericMobileHandle(e)
            this.handleStart(realX, realY, rx, ry)
        })
        this.element.addEventListener('touchmove', e => {
            e.stopPropagation()
            const {sx: realX, sy: realY} = genericMobileHandle(e)
            this.handleMove(realX, realY)
        })
        this.element.addEventListener('touchend', this.endDraw.bind(this))
        this.element.addEventListener('touchcancel', this.endDraw.bind(this))
    }
}