import { cloneDeep } from 'lodash'
import { onUnmounted, onActivated, onDeactivated, getCurrentInstance } from 'vue'

interface Event {
    readonly uid: string
    cb: (data: any) => void
    alive?: boolean
    del?: boolean
    only?: boolean
}

class EventBus {
    protected eventMap: { [key: string]: Array<Event> }
    constructor() {
        this.eventMap = {}
    }
    // 发布事件
    emit(eventName: string, ...args: Array<any>) {
        let eventList = this.eventMap[eventName]
        if (eventList) {
            let delIndexs: Array<number> = []
            for (let i = 0; i < eventList.length; i++) {
                let { cb, del = false, only = false, alive } = eventList[i]
                if (!alive) {
                    continue
                }
                cb(...(args as [any]))
                if (del) {
                    delIndexs.push(i)
                }
                if (only) {
                    break
                }
            }
            if (delIndexs.length) {
                this.eventMap[eventName] = eventList.filter((_, i) => {
                    return !delIndexs.includes(i)
                })
            }
        } else if (import.meta.env.MODE === 'development') {
            console.warn(eventName + ' not found!')
        }
    }
    // 订阅事件
    on(eventName: string, opt: Event) {
        if (!this.eventMap[eventName]) {
            this.eventMap[eventName] = []
        }
        let uid = opt.uid
        let index = this.eventMap[eventName].findIndex((item) => {
            return uid == item.uid
        })
        if (index >= 0) {
            this.eventMap[eventName].splice(index, 1, opt)
            return
        }
        this.eventMap[eventName].unshift(opt)
    }

    // 取消订阅时间
    off(eventName: string, uid: string) {
        if (this.eventMap[eventName]) {
            let index = this.eventMap[eventName].findIndex((item) => {
                return uid == item.uid
            })
            if (index >= 0) {
                this.eventMap[eventName].splice(index, 1)
                if (!this.eventMap[eventName].length) {
                    delete this.eventMap[eventName]
                }
            }
        }
        // console.log(this.eventMap)
    }

    // 只订阅一次
    one(eventName: string, opt: Event) {
        opt.del = true
        return this.on(eventName, opt)
    }

    // 激活
    enable(eventName: string, uid: string) {
        if (this.eventMap[eventName]) {
            let index = this.eventMap[eventName].findIndex((item) => {
                return uid == item.uid
            })
            if (index == 0) {
                this.eventMap[eventName][index].alive = true
            } else if (index > 0) {
                const newEvent = cloneDeep(this.eventMap[eventName][index])
                this.eventMap[eventName].splice(index, 1)
                newEvent.alive = true
                this.eventMap[eventName].unshift(newEvent)
            }
        }
    }

    // 禁用
    disable(eventName: string, uid: string, alive: boolean = false) {
        if (this.eventMap[eventName]) {
            let index = this.eventMap[eventName].findIndex((item) => {
                return uid == item.uid
            })
            if (index >= 0) {
                this.eventMap[eventName][index].alive = alive
            }
        }
    }
}
const eventBus = new EventBus()

export function useBusOn(eventName: string, cb: (value: any) => void, opt?: { del?: boolean; only?: boolean; alive?: boolean }) {
    const currentInstance = getCurrentInstance()
    const _opt = {
        uid: String(currentInstance?.uid),
        cb: cb as (value: any) => void,
        alive: true,
        ...opt
    }
    eventBus.on(eventName, _opt)
    onActivated(() => {
        eventBus.on(eventName, _opt)
    })
    function off() {
        eventBus.off(eventName, _opt.uid)
    }
    onUnmounted(() => off())
    onDeactivated(() => {
        off()
    })
    return {
        off,
        enable: () => {
            eventBus.enable(eventName, _opt.uid)
        },
        disable: () => {
            eventBus.disable(eventName, _opt.uid)
        }
    }
}

export function useBusOne(eventName: string, cb: (value: any) => void, opt?: { del?: boolean; only?: boolean }) {
    let _opt = { ...opt, del: true }
    return useBusOn(eventName, cb, _opt)
}

export function useBusOff(eventName: string) {
    const currentInstance = getCurrentInstance()
    const uid = String(currentInstance?.uid)
    eventBus.off(eventName, uid)
}

export function useBusEmit(eventName: string, ...args: Array<any>) {
    eventBus.emit(eventName, ...(args as Array<any>))
}
