import Day from '@/utils/dayjs'

/**
 * 通用的格式化时间
 * @param input {String | Date | Number | null} 可以被传入Date构造函数的任意数据
 * @param until {String} 为all、day、min中的任何一个，默认为all，表示格式化到哪一位
 * @param defaultVal {String} 当传入数据不合法时的默认值，默认为'--’
 * @return String 格式化之后的字符串
 */
export const formatDate = (input: Date | string | number | null, until = 'all', defaultVal: string = '--'): string => {
    // @ts-ignore
    if (input === null || input === 0 || input < 0) return defaultVal
    const date = new Date(input)
    const year = date.getFullYear()
    const month = date.getMonth() + 1
    const day = date.getDate()
    const hour = date.getHours()
    const min = date.getMinutes()
    const sec = date.getSeconds()
    if (until === 'day') return `${year}-${month.toString().padStart(2, '0')}-${day.toString().padStart(2, '0')}`
    if (until === 'min') {
        return `${year}-${month.toString().padStart(2, '0')}-${day.toString().padStart(2, '0')} ${hour.toString().padStart(2, '0')}:${min.toString().padStart(2, '0')}`
    }
    return `${year}-${month.toString().padStart(2, '0')}-${day.toString().padStart(2, '0')} ${hour.toString().padStart(2, '0')}:${min.toString().padStart(2, '0')}:${sec.toString().padStart(2, '0')}`
}

/**
 * 加上年月日时分秒的中国本土化的格式化时间
 * @param input {String | Date | Number | null} 可以被传入Date构造函数的任意数据
 * @param until {String} 为all、day、min中的任何一个，默认为all，表示格式化到哪一位
 * @param defaultVal {String} 当传入数据不合法时的默认值，默认为'--’
 * @return String 格式化之后的字符串
 */
export const formatDateZh = (input: Date | string | number | null, until = 'all', defaultVal: string = '--'): string => {
    // @ts-ignore
    if (input === null || input === 0 || input < 0) return defaultVal
    const date = new Date(input)
    const year = date.getFullYear()
    const month = date.getMonth() + 1
    const day = date.getDate()
    const hour = date.getHours()
    const min = date.getMinutes()
    const sec = date.getSeconds()
    if (until === 'day') return `${year}年${month.toString().padStart(2, '0')}月${day.toString().padStart(2, '0')}日`
    if (until === 'min') {
        return `${year}年${month.toString().padStart(2, '0')}月${day.toString().padStart(2, '0')}日 ${hour.toString().padStart(2, '0')}:${min.toString().padStart(2, '0')}`
    }
    return `${year}年${month.toString().padStart(2, '0')}月${day.toString().padStart(2, '0')}日 ${hour.toString().padStart(2, '0')}:${min
        .toString()
        .padStart(2, '0')}:${sec.toString().padStart(2, '0')}`
}

/**
 * 时间戳格式化
 * 1715072168340 => 2024-05-07 16:56:08
 */
export const timeStampFormat = (timeStamp: string | number, opt?: { format?: string }): string => {
    if (!timeStamp) {
        return ''
    }
    timeStamp = parseInt(String(timeStamp))
    const format = opt?.format || 'YYYY-MM-DD HH:mm:ss'
    return Day(timeStamp).format(format)
}
