import {utils, WorkBook, writeFile} from "xlsx"

/**
 * 将某个html表格直接导出
 * @param DOMElement {HTMLTableElement} 待导出的html表格元素
 * @param fileName {String} 导出的excel文件名称
 */
export function exportLocallyExcelFunc(DOMElement: HTMLTableElement, fileName: string) {
    const worksheet = utils.table_to_sheet(DOMElement, {})
    const workbook: WorkBook = {
        SheetNames: [fileName],
        Sheets: {
            [fileName]: worksheet
        }
    }
    writeFile(workbook, fileName + '.xlsx', {
        type: 'file',
        bookType: 'xlsx'
    })
}

/**
 * 导出为包含多个表格的excel文件
 * @param fileName {String} 导出的excel文件名
 * @param group {{[key: string] : T[][]}} 为一个值为二维数组的键值对，表示表格名以及其内部数据。数组是一行一行数据的集合。二维数组，代表每一行每一列的数据，比如[[1,2], [3,4]]表示第一行第一列数据为1，第一行第二列数据为2，以此类推
 */
export function exportMultipleAOA<T>(fileName: string, group: { [key: string]: T[][] }) {
    const workSheetsGrouping = Object.entries(group).reduce((q, w) => {
        // @ts-ignore
        q[w[0]] = utils.aoa_to_sheet(w[1], {})
        return q
    }, {})
    const workbook: WorkBook = {
        SheetNames: Object.keys(group),
        Sheets: workSheetsGrouping
    }
    writeFile(workbook, fileName + '.xlsx', {
        type: 'file',
        bookType: 'xlsx'
    })
}

/**
 * 导出只包含一个表格的excel文件
 * @param data {T[][]} 二维数组，代表每一行每一列的数据，比如[[1,2], [3,4]]表示第一行第一列数据为1，第一行第二列数据为2，以此类推
 * @param fileName {String} 导出的excel文件名
 */
export function exportAOA<T>(data: T[][], fileName: string) {
    const worksheet = utils.aoa_to_sheet(data, {})
    worksheet['A1'].s = {font: {color: {rgb: "FF0000"}}}
    const workbook: WorkBook = utils.book_new()
    utils.book_append_sheet(workbook, worksheet, 'Sheet1')
    writeFile(workbook, fileName + '.xlsx', {
        type: 'file',
        bookType: 'xlsx'
    })
}