use crate::config::config::STATIC_WS_PWD;
use crate::deport::handle_ws_msg;
use crate::events::{register_client, ClientMessage, Event};
use crate::handles::handshake::handle_websocket_handshake;
use crate::handles::heartbeat::heart_resp;
use crate::handles::online_users_update::send_online_users_and_send;
use crate::utils::json_utils::parse_message;
use dashmap::DashMap;
use futures::{SinkExt, StreamExt};
use lazy_static::lazy_static;
use std::time::{Duration, Instant, SystemTime, UNIX_EPOCH};
use tokio::sync::mpsc::{UnboundedReceiver, UnboundedSender};
use tokio::sync::watch;
use tokio::time;
use tungstenite::{Error, Message};

lazy_static! {
    #[derive(Debug)]
    pub static ref ONLINE_USERS: DashMap<String, String> = DashMap::new();
}

pub(crate) async fn handle_client(
    stream: tokio::net::TcpStream,
    event_sender: UnboundedSender<Event>,
    center_to_client_sender: UnboundedSender<ClientMessage>,
    mut center_to_client_receiver: UnboundedReceiver<ClientMessage>,
) -> Result<(), Error> {
    let must_existed_params = ["deviceID", "fromID", "wsPwd"];
    let mut connection_params = None;

    let ws_stream = match handle_websocket_handshake(
        stream,
        &must_existed_params,
        STATIC_WS_PWD,
        &mut connection_params,
    )
    .await
    {
        Ok(ws) => ws,
        Err(e) => {
            println!("WebSocket握手失败: {:?}", e);
            return Ok(());
        }
    };

    let (mut sender, mut receiver) = ws_stream.split();
    let mut last_heartbeat_time = Instant::now(); // 包装为 Arc<Mutex<Instant>>

    if let Some(params) = connection_params {
        if let Some(from_id) = params.get("fromID") {
            let from_id = from_id.clone();
            // 获取当前时间戳
            let connection_time = SystemTime::now()
                .duration_since(UNIX_EPOCH)
                .expect("时间获取失败")
                .as_nanos();

            // 注册客户端到事件中心
            register_client((from_id.clone(), connection_time), center_to_client_sender).await;

            // 发送新连接事件
            match event_sender.send(Event::NewConnection(from_id.clone(), params)) {
                Ok(_) => println!("新连接事件发送成功"),
                Err(e) => println!("发送新连接事件时出错: {:?}", e),
            }

            println!("用户 {} 已连接", from_id);

            // 使用 watch 通道来发送关闭信号
            let (close_tx, mut close_rx) = watch::channel(false);

            // 启动事件中心调度处理任务
            let event_task = tokio::spawn({
                let from_id = from_id.clone();
                async move {
                    while let Some(msg) = center_to_client_receiver.recv().await {
                        println!("消息中心：==> 收到消息 {:?}", &msg);
                        match msg {
                            ClientMessage::CmdUpdateOnlineUsers => {
                                println!(
                                    "消息中心：==> 收到 CmdUpdateOnlineUsers 消息 发送给 {}",
                                    &from_id
                                );
                                if let Err(e) =
                                    send_online_users_and_send(&mut sender, &from_id).await
                                {
                                    println!("处理在线用户列表出错了：{:?}", e);
                                    break;
                                }
                            }
                            ClientMessage::SendClientMessage(from_id, client_message, close) => {
                                let real_user_id = from_id.0;
                                println!(
                                    "消息中心：==> 收到消息 {:?} 发送给 {}",
                                    &client_message,
                                    real_user_id.clone()
                                );
                                if let Err(e) = sender.send(Message::text(client_message)).await {
                                    println!("发送给用户id {} 独立消息失败：{:?}", real_user_id, e);
                                    break;
                                }
                                if close {
                                    // 通知外层循环关闭
                                    match close_tx.send(true) {
                                        Ok(_) => println!("关闭信号发送成功"),
                                        Err(e) => println!("发送关闭信号时出错: {:?}", e),
                                    }
                                    println!("发送给用户id {} 要求关闭连接", real_user_id);
                                    break;
                                }
                            }
                        }
                    }
                }
            });

            // 处理 WebSocket 消息和心跳
            loop {
                tokio::select! {
                    // 处理消息接收
                    maybe_msg = receiver.next() => {
                        if let Some(Ok(msg)) = maybe_msg {
                            if msg.is_text() {
                                let message = msg.to_text()?;
                                match parse_message(message) {
                                    Ok(data) => {
                                        match data.msg_type.as_str() {
                                            "Heart" => {
                                                println!("收到客户端心跳消息 {:?}", &data);
                                                last_heartbeat_time = Instant::now();
                                                if let Ok(json_str) = heart_resp(&from_id) {
                                                    if let Err(e) = event_sender.send(Event::SendClientMessage((from_id.clone(), connection_time), json_str, false)){
                                                        println!("处理心跳消息出错了：{:?}", e)
                                                    } else {
                                                        println!("处理心跳消息成功")
                                                    }
                                                }
                                            },
                                            _ => {
                                                let from_id_clone = from_id.clone();
                                                let connection_time_clone = connection_time.clone();
                                                handle_ws_msg(&data, from_id_clone, &event_sender, connection_time_clone, message).await;
                                            }
                                        }
                                    }
                                    Err(e) => {
                                        println!("解析JSON数据出错: {}", e);
                                    }
                                }
                            }
                        } else {
                            // 断开连接之后直接移除即可
                            match event_sender.send(Event::CloseConnection((from_id.clone(), connection_time))) {
                                Ok(_) => println!("关闭连接事件发送成功"),
                                Err(e) => println!("发送关闭连接事件时出错: {:?}", e),
                            }
                            break; // 客户端断开连接
                        }
                    }
                    // 处理心跳超时
                    _ = time::sleep_until(tokio::time::Instant::from(last_heartbeat_time + Duration::from_secs(20))) => {
                        println!("用户id-{} 20秒内没有发送心跳，挂断连接", from_id);
                        // 发送关闭连接事件
                        match event_sender.send(Event::CloseConnection((from_id.clone(), connection_time))) {
                            Ok(_) => println!("关闭连接事件发送成功"),
                            Err(e) => println!("发送关闭连接事件时出错: {:?}", e),
                        }
                        break;
                    }
                    // 监听关闭通知
                    _ = close_rx.changed() => {
                        if *close_rx.borrow() {
                            println!("收到关闭通知，退出循环");
                            break;
                        }
                    }
                }
            }

            // 发送关闭连接事件
            if let Err(e) =
                event_sender.send(Event::CloseConnection((from_id.clone(), connection_time)))
            {
                println!("发送关闭消息到事件中心失败！ 原因： {:?}", e);
            } else {
                println!("发送关闭消息到事件中心成功！");
            }

            // 等待事件中心调度任务结束
            if let Err(e) = event_task.await {
                println!("事件中心调度任务出错: {:?}", e);
            }
        }
    } else {
        println!("无法获取连接参数");
    }

    Ok(())
}
