use crate::events::{Event, CLIENT_SENDERS};
use crate::handles::handle_agora_call::handle_agora_call;
use crate::handles::online_users_update::{online_messages, send_online_users_resp};
use crate::typing::message_typed::ClientMessageData;
use tokio::sync::mpsc::UnboundedSender;
use crate::client::ONLINE_USERS;
use crate::handles::patch_record::send_request;

const CHAT_TYPE_MESSAGE: [&str; 5] = ["ChatTxt", "ChatImage", "ChatFile", "ChatVideo", "ChatAudio"];

pub async fn handle_ws_msg(
    client_message_data: &ClientMessageData,
    from_id: String,
    event_sender: &UnboundedSender<Event>,
    connection_time: u128,
    raw_msg: &str
) {
    let msg_type = client_message_data.msg_type.clone();
    match msg_type.as_str() {
        "GetCompanyUserList" => {
            println!("收到客户端获取在线用户列表 {}", &from_id);
            // 需要主动发送在线人员更新事件
            if let Ok(online_users_json) = send_online_users_resp().await {
                if let Ok(formatted_json) = online_messages(online_users_json, from_id.clone()) {
                    if let Err(e) = event_sender.send(Event::SendClientMessage(
                        (from_id.clone(), connection_time),
                        formatted_json,
                        false,
                    )) {
                        println!(
                            "客户端要求更新在线列表 {}发送在线用户列表失败 {:?}",
                            &from_id, e
                        );
                    } else {
                        println!("客户端要求更新在线列表 {}发送在线用户列表成功", &from_id);
                    }
                }
            }
        }
        // 通话类消息直接托管给对应句柄即可
        "Call" | "CancelCall" | "Refuse" | "EndMeeting" | "Hangup" | "Connect" | "Mute"
        | "MuteAll" | "KickOut" | "MuteSelf" | "UnMuteSelf" => {
            handle_agora_call(
                &client_message_data,
                &from_id,
                &event_sender,
                &connection_time,
            )
            .await;
        }
        // 针对其余消息类型，直接根据fromID -> toID规则发送即可
        _ => {
            let to_id = &client_message_data.to_id;
            let msg_json = serde_json::json!({
                "msgType": msg_type,
                "msgData": client_message_data.msg_data,
                "fromID": from_id,
                "fromName": client_message_data.from_name,
                "toID": to_id
            }).to_string();

            // 如果包含chat-msg，则发送额外消息，开启一个新协程
            if CHAT_TYPE_MESSAGE.contains(&client_message_data.msg_type.as_str()) {
                let from_id_c = from_id.to_string();
                let to_id_c = to_id.to_string();
                let r_msg = raw_msg.to_string();
                tokio::spawn(async move{
                    if let Err(e) = send_request(from_id_c.as_str(), to_id_c.as_str(), r_msg.as_str()).await {
                        println!("发送upload-patch请求失败: {:?}", e);
                    } else {
                        println!("发送upload-patch请求成功");
                    }
                });
            }

            println!("收到客户端消息 类型: {} 来自: {} 发送给: {}", msg_type, from_id, to_id);

            if to_id == "-2" {
                // 广播消息给同频道的所有在线用户
                if let Some(user_data) = ONLINE_USERS.get(&from_id) {
                    let channel_id = user_data.split(',').nth(1).unwrap_or("").to_string();
                    println!("准备广播消息到频道: {} 来自用户: {}", channel_id, from_id);
                    tokio::spawn({
                        let event_sender = event_sender.clone();
                        let msg_json = msg_json.clone();
                        async move {
                            for entry in ONLINE_USERS.iter() {
                                let target_id = entry.key();
                                let target_data = entry.value();
                                if target_data.split(',').nth(1).unwrap_or("") == channel_id {
                                    if let Some(sender) = CLIENT_SENDERS.iter()
                                        .find(|e| e.key().0 == *target_id)
                                        .map(|e| e.key().clone()) {
                                        println!("发送广播消息给用户: {} 内容: {}", target_id, msg_json);
                                        if let Err(e) = event_sender.send(Event::SendClientMessage(
                                            sender,
                                            msg_json.clone(),
                                            false,
                                        )) {
                                            println!("发送广播消息给用户: {} 失败: {:?}", target_id, e);
                                        } else {
                                            println!("发送广播消息给用户: {} 成功", target_id);
                                        }
                                    } else {
                                        println!("未找到用户: {} 的sender，无法发送广播消息", target_id);
                                    }
                                }
                            }
                        }
                    });
                } else {
                    println!("用户: {} 不在线或无频道信息，无法广播", from_id);
                }
            } else {
                // 单播消息给指定用户
                if ONLINE_USERS.contains_key(to_id) {
                    if let Some(sender) = CLIENT_SENDERS.iter()
                        .find(|e| e.key().0 == *to_id)
                        .map(|e| e.key().clone()) {
                        println!("发送单播消息给用户: {} 内容: {}", to_id, msg_json);
                        if let Err(e) = event_sender.send(Event::SendClientMessage(
                            sender,
                            msg_json,
                            false,
                        )) {
                            println!("发送单播消息给用户: {} 失败: {:?}", to_id, e);
                        } else {
                            println!("发送单播消息给用户: {} 成功", to_id);
                        }
                    } else {
                        println!("未找到用户: {} 的sender，无法发送单播消息", to_id);
                    }
                } else {
                    println!("目标用户: {} 不在线，无法发送单播消息", to_id);
                }
            }
        }
    }
}
