use crate::handles::redis::{insert_this_connection, remove_this_connection};
use crate::typing::message_typed::OtherLoginMessageData;
use dashmap::DashMap;
use lazy_static::lazy_static;
use serde_json::Value::Null;
use std::collections::HashMap;
use tokio::sync::mpsc;
use urlencoding::decode;
use crate::client::ONLINE_USERS;

// 定义事件类型
#[derive(Debug)]
pub enum Event {
    NewConnection(String, HashMap<String, String>),
    CloseConnection((String, u128)),
    SendClientMessage((String, u128), String, bool),
}

// 定义事件中心发送给客户端的消息类型
#[derive(Debug)]
pub enum ClientMessage {
    CmdUpdateOnlineUsers,
    SendClientMessage((String, u128), String, bool),
}

lazy_static! {
    #[derive(Debug)]
    pub static ref CLIENT_SENDERS: DashMap<(String, u128), mpsc::UnboundedSender<ClientMessage>> =
        DashMap::new();
}

// 注册客户端的发送者
pub async fn register_client(
    from_id: (String, u128),
    center_to_client_sender: mpsc::UnboundedSender<ClientMessage>,
) {
    close_existing_connection(&from_id, true).await;
    println!("注册用户 {:?} 前数量 {}", &from_id, CLIENT_SENDERS.len());
    CLIENT_SENDERS.insert(from_id.clone(), center_to_client_sender);
    println!("注册用户 {:?} 后数量 {}", from_id, CLIENT_SENDERS.len());
}

// 关闭之前绑定的 WebSocket 连接
// 关闭之前绑定的 WebSocket 连接
pub async fn close_existing_connection(from_id: &(String, u128), close_old: bool) {
    let (real_user_id, current_time) = (&from_id.0, from_id.1);

    if close_old {
        // 关闭旧连接：找到所有连接时间小于当前时间的连接
        let mut keys_to_remove = Vec::new();
        for entry in CLIENT_SENDERS.iter() {
            println!("当前key {:?}", entry.key());
            if entry.key().0 == real_user_id.to_string() && entry.key().1 < current_time {
                keys_to_remove.push(entry.key().clone());
            }
        }

        for key in keys_to_remove {
            if let Some((_, sender)) = CLIENT_SENDERS.remove(&key) {
                let other_login_msg = OtherLoginMessageData {
                    msg_type: "OtherLoginIn".to_string(),
                    from_id: "0".to_string(),
                    from_name: "Server".to_string(),
                    msg_data: Null,
                    to_id: key.0.clone(),
                };
                if let Err(e) = sender.send(ClientMessage::SendClientMessage(key.clone(), serde_json::to_string(&other_login_msg).unwrap(), true)) {
                    println!("通知客户端 {:?} 关闭连接失败: {:?}", key, e);
                } else {
                    println!("通知客户端 {:?} 关闭连接成功", key);
                }
            }
            {
                ONLINE_USERS.remove(&key.0.clone());
                println!("成功从全局变量中移除用户id: {:?} 的信息", from_id);
            }
            tokio::spawn(async move {
                if let Err(e) = remove_this_connection(&key.0).await {
                    println!("从 Redis 中移除用户信息时出错: {}", e);
                }
            });
        }
    } else {
        // 关闭当前连接
        if let Some((_, sender)) = CLIENT_SENDERS.remove(from_id) {
            let other_login_msg = OtherLoginMessageData {
                msg_type: "OtherLoginIn".to_string(),
                from_id: "0".to_string(),
                from_name: "Server".to_string(),
                msg_data: Null,
                to_id: real_user_id.clone(),
            };
            if let Err(e) = sender.send(ClientMessage::SendClientMessage(from_id.clone(), serde_json::to_string(&other_login_msg).unwrap(), true)) {
                println!("通知客户端 {:?} 关闭连接失败: {:?}", from_id, e);
            } else {
                println!("通知客户端 {:?} 关闭连接成功", from_id);
            }
        }
        {
            ONLINE_USERS.remove(&real_user_id.clone());
            println!("成功从全局变量中移除用户id: {:?} 的信息", from_id);
        }
        let user_id_clone = real_user_id.clone();
        tokio::spawn(async move{
            if let Err(e) = remove_this_connection(user_id_clone.as_str()).await {
                println!("从 Redis 中移除用户信息时出错: {}", e);
            }
        });
    }
}

// 创建连接参数集合
fn create_connection_params(from_id: &str, params: &HashMap<String, String>,) -> String {
    let device_id = params.get("deviceID").cloned().unwrap_or("".to_string());
    let from_name = params.get("fromName").cloned().unwrap_or("".to_string());
    let has_mike = if let Some(has_mike) = params.get("hasMike") {
        has_mike.to_string()
    } else {
        "0".to_string()
    };
    let has_camera = if let Some(has_camera) = params.get("hasCamera") {
        has_camera.to_string()
    } else {
        "0".to_string()
    };
    let user_call_group = if let Some(user_call_group) = params.get("userCallGroup") {
        user_call_group.to_string()
    } else {
        "0".to_string()
    };
    let from_name_decoded = match decode(from_name.as_str()) {
        Ok(decoded) => decoded.to_string(),
        Err(_) => from_name,
    };
    // 按照结构体OnlineUserMessage的格式构造用户信息字符串，用逗号拼接即可
    // callState,channelID,deviceID,fromID,hasCamera,hasMike,isHost,userCallGroup,fromName
    format!(
        "{},{},{},{},{},{},0,{},{}",
        "idle", "", device_id, from_id, has_camera, has_mike, user_call_group, from_name_decoded
    )
}

// 处理事件的任务
pub async fn handle_events(mut receiver: mpsc::UnboundedReceiver<Event>) {
    while let Some(event) = receiver.recv().await {
        match event {
            Event::NewConnection(from_id, params) => {
                println!("有新的连接 用户id {} 更新在线用户列表事件触发", from_id);
                // 将该用户的信息插入到 Redis 中
                let connection_p = create_connection_params(
                    &from_id, &params
                );
                {
                    // 获取写锁以进行写操作
                    ONLINE_USERS.insert(from_id.to_string(), connection_p.clone());
                    println!("成功将用户id: {} 的信息插入到 ONLINE_USERS 中", from_id);
                }
                tokio::spawn(async move {
                    if let Err(e) = insert_this_connection(&from_id, connection_p).await {
                        println!("将用户信息插入到 Redis 中时出错: {}", e);
                    }
                });
                tokio::spawn(async move {
                    notify_all_clients_to_update_online_users().await;
                });
            }
            Event::CloseConnection(from_id) => {
                println!("有关闭的连接 用户id {:?} 更新在线用户列表事件触发", from_id);
                close_existing_connection(&from_id, false).await;
                tokio::spawn(async move {
                    notify_all_clients_to_update_online_users().await;
                });
            }
            Event::SendClientMessage(from_id, msg, close) => {
                println!("事件中心代发消息 ===>> 发送消息 {:?} 到客户端 {:?}", msg, from_id);
                let target_sender = CLIENT_SENDERS.get(&from_id).unwrap();
                println!("成功获取到sender标记");
                if let Err(e) = target_sender
                    .send(ClientMessage::SendClientMessage(from_id, msg, close))

                {
                    println!("通知对应sender发送ws消息失败 {}", e);
                } else {
                    println!("通知对应sender发送ws消息成功");
                }
            }
        }
    }
}

// 通知所有客户端线程发送 CmdUpdateOnlineUsers 消息
pub async fn notify_all_clients_to_update_online_users() {
    println!(
        "尝试发送在新用户更新消息体，目前总人数为：{}",
        CLIENT_SENDERS.len()
    );
    for entry in CLIENT_SENDERS.iter() {
        let sender: &mpsc::UnboundedSender<ClientMessage> = entry.value();
        let from_id = entry.key();
        if let Err(e) = sender.send(ClientMessage::CmdUpdateOnlineUsers) {
            println!("通知客户端 {:?} 更新在线用户列表失败: {:?}", from_id, e);
        } else {
            println!("通知客户端 {:?} 更新在线用户列表成功 ===> $$$", from_id);
        }
    }
}
