use crate::client::ONLINE_USERS;
use crate::events::{notify_all_clients_to_update_online_users, Event, CLIENT_SENDERS};
use crate::handles::handshake::HOST_ENABLED_ID_SET;
use crate::handles::redis::update_client_redis_data;
use crate::typing::message_typed::ClientMessageData;
use crate::utils::json_utils::get_current_timestamp;
use dashmap::DashMap;
use lazy_static::lazy_static;
use std::time::Duration;
use tokio::sync::mpsc::{self, UnboundedSender};
use tokio::time::sleep;

lazy_static! {
    pub static ref refuse_procedure_map: DashMap<String, UnboundedSender<()>> = DashMap::new();
    pub static ref channel_hangup_procedure_map: DashMap<String, UnboundedSender<()>> = DashMap::new();
}

// 用户状态字段的索引
const STATUS_IDX: usize = 0;
const CHANNEL_IDX: usize = 1;
const HOST_IDX: usize = 6;

// 异步更新 Redis 数据
fn update_redis_async(user_id: String, data: String) {
    tokio::spawn(async move {
        if let Err(e) = update_client_redis_data(&user_id, data).await {
            println!("更新redis数据失败：{:?} 用户id {}", e, user_id);
        } else {
            println!("更新redis数据成功");
        }
    });
}

// 更新用户状态并同步到 Redis 和 ONLINE_USERS
fn update_user_status(user_id: &str, status: &str, channel_id: &str, is_host: bool) -> String {
    let mut user_data = ONLINE_USERS
        .get(user_id)
        .map(|v| v.split(',').map(String::from).collect())
        .unwrap_or(vec!["idle".into(), "".into(), "".into(), "".into(), "".into(), "".into(), "0".into()]);
    user_data[STATUS_IDX] = status.into();
    user_data[CHANNEL_IDX] = channel_id.into();
    user_data[HOST_IDX] = if is_host { "1".into() } else { "0".into() };
    let joined = user_data.join(",");
    ONLINE_USERS.insert(user_id.to_string(), joined.clone());
    joined
}

// 发送消息的通用函数
async fn send_inside_message(
    target_sender: &(String, u128),
    event_sender: &UnboundedSender<Event>,
    json_message: String,
    from_id: &String,
) {
    println!("发送给用户id {} 的消息 {}", from_id, json_message);
    if let Err(e) = event_sender.send(Event::SendClientMessage(target_sender.clone(), json_message, false)) {
        println!("发送给用户id {} 独立消息失败：{:?}", from_id, e);
    } else {
        println!("发送给用户id {} 独立消息成功", from_id);
    }
}

// 发送错误消息
async fn send_error_message(
    target_sender: &(String, u128),
    event_sender: &UnboundedSender<Event>,
    from_id: &String,
    error_msg: &str,
) {
    let json = serde_json::json!({
        "msgType": "Error",
        "msgData": error_msg,
        "fromID": "0",
        "fromName": "Server",
        "toID": from_id
    }).to_string();
    if let Err(e) = event_sender.send(Event::SendClientMessage(target_sender.clone(), json, false)) {
        println!("发送给用户id {} 独立消息失败：{:?}", from_id, e);
    } else {
        println!("发送给用户id {} 独立消息成功", from_id);
    }
}

// 获取指定 channel_id 的用户 ID 列表
fn get_users_by_channel(channel_id: &str) -> Vec<String> {
    ONLINE_USERS
        .iter()
        .filter(|e| e.value().split(',').nth(CHANNEL_IDX).unwrap_or("") == channel_id)
        .map(|e| e.key().clone())
        .collect()
}

fn hangup_last_user(user_id: String, event_sender_clone: UnboundedSender<Event>, channel_id_clone: String, mut cancel_rx: mpsc::UnboundedReceiver<()>) {
    tokio::spawn(async move {
        tokio::select! {
            _ = sleep(Duration::from_secs(15)) => {
                println!("20s内如果该频道与会人数仍为1，则对剩下来的用户发送CmdHangup指令 {}", user_id);
                if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                    let joined = update_user_status(&user_id, "idle", "", false);
                    update_redis_async(user_id.clone(), joined);
                    send_inside_message(&sender, &event_sender_clone, serde_json::json!({
                        "msgType": "CmdHangup",
                        "fromID": "0",
                        "fromName": "Server",
                        "toID": user_id,
                        "msgData": {"channelId": "", "rtcToken": ""}
                    }).to_string(), &user_id).await;
                    tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
                }
            }
            _ = cancel_rx.recv() => println!("收到取消信号，有新的人员加入到会议，会议频道号 {} 结束线程", channel_id_clone),
        }
    });
}

pub async fn handle_agora_call(
    client_message_data: &ClientMessageData,
    from_id: &String,
    event_sender: &UnboundedSender<Event>,
    connection_time: &u128,
) {
    let target_sender_which = (from_id.to_string(), connection_time.clone());

    match client_message_data.msg_type.as_str() {
        "CancelCall" => {
            let calling_to_id = &client_message_data.to_id;
            println!("收到客户端取消呼叫 取消呼叫组: {} 呼叫方id {}", calling_to_id, from_id);
            send_inside_message(&target_sender_which, event_sender, serde_json::json!({
                "msgType": "CmdCancelCall",
                "fromID": "0",
                "fromName": "Server",
                "toID": from_id,
                "msgData": {"channelId": "", "rtcToken": ""}
            }).to_string(), from_id).await;

            if let Some(user_status) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data = user_status.split(',').map(String::from).collect::<Vec<_>>();
                if user_data[CHANNEL_IDX].is_empty() {
                    let joined = update_user_status(from_id, "idle", "", false);
                    update_redis_async(from_id.clone(), joined);
                    tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
                    return;
                }

                let to_hangup_ids: Vec<String> = calling_to_id.split(',').map(String::from).collect();
                for to_id in to_hangup_ids {
                    if let Some(sender) = refuse_procedure_map.get(&to_id) {
                        sender.send(()).ok();
                    }
                    if let Some(other_data) = ONLINE_USERS.get(&to_id).map(|v| v.clone()) {
                        let other_data_vec = other_data.split(',').map(String::from).collect::<Vec<_>>();
                        if other_data_vec[CHANNEL_IDX] == user_data[CHANNEL_IDX] {
                            if let Some(target_sender) = CLIENT_SENDERS.iter().find(|entry| entry.key().0 == *calling_to_id).map(|entry| entry.key().clone()) {
                                send_inside_message(&target_sender, event_sender, serde_json::json!({
                                    "msgType": "CmdHangup",
                                    "fromID": "0",
                                    "fromName": "Server",
                                    "toID": to_id,
                                    "msgData": {"channelId": "", "rtcToken": ""}
                                }).to_string(), &to_id).await;
                                let joined = update_user_status(&to_id, "idle", "", false);
                                update_redis_async(to_id, joined);
                            } else {
                                println!("未找到对应的sender");
                            }
                        }
                    }
                }
                let joined = update_user_status(from_id, "idle", "", false);
                update_redis_async(from_id.clone(), joined);
                tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
            }
        }

        "Call" => {
            let calling_to_id = &client_message_data.to_id;
            println!("收到客户端呼叫 呼叫方id {} 呼叫组id {}", from_id, calling_to_id);
            if calling_to_id.is_empty() {
                send_error_message(&target_sender_which, event_sender, from_id, "请指定呼叫对象").await;
                return;
            }
            if calling_to_id == from_id {
                send_error_message(&target_sender_which, event_sender, from_id, "不能给自己打电话").await;
                return;
            }

            if let Some(user_status) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data = user_status.split(',').map(String::from).collect::<Vec<_>>();
                let channel_id = if user_data[CHANNEL_IDX].is_empty() {
                    format!("{}-{}", from_id, get_current_timestamp())
                } else {
                    user_data[CHANNEL_IDX].clone()
                };
                println!("当前用户channelId {} 呼叫方id集合是 {:?}", channel_id, calling_to_id.split(',').collect::<Vec<_>>());
                if let Some(sender) = channel_hangup_procedure_map.get(&channel_id) { sender.send(()).ok(); }

                let mut refuse_users = Vec::new();
                for to_id in calling_to_id.split(',') {
                    if !ONLINE_USERS.contains_key(to_id) {
                        send_error_message(&target_sender_which, event_sender, from_id, "对方不在线").await;
                        continue;
                    }
                    if let Some(to_status) = ONLINE_USERS.get(to_id).map(|v| v.clone()) {
                        let to_data = to_status.split(',').map(String::from).collect::<Vec<_>>();
                        if to_data[STATUS_IDX] != "idle" {
                            send_error_message(&target_sender_which, event_sender, from_id, "对方正在呼叫中").await;
                            continue;
                        }
                        if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == to_id).map(|e| e.key().clone()) {
                            send_inside_message(&sender, event_sender, serde_json::json!({
                                "msgType": "Call",
                                "fromID": from_id,
                                "fromName": client_message_data.from_name,
                                "toID": to_id,
                                "msgData": {"channelId": channel_id, "rtcToken": ""}
                            }).to_string(), &to_id.to_string()).await;
                            let joined = update_user_status(to_id, "callin", &channel_id, false);
                            update_redis_async(to_id.to_string(), joined);
                            refuse_users.push(to_id.to_string());
                        } else {
                            println!("找不到对应的sender，无法发送客户端消息！");
                        }
                    }
                }

                if !refuse_users.is_empty() {
                    let status = if user_data[STATUS_IDX] == "calling" { "calling" } else { "callout" };
                    send_inside_message(&target_sender_which, event_sender, serde_json::json!({
                        "msgType": "CmdCall",
                        "fromID": calling_to_id,
                        "fromName": client_message_data.from_name,
                        "toID": from_id,
                        "msgData": {"channelId": channel_id, "rtcToken": ""}
                    }).to_string(), from_id).await;
                    let joined = update_user_status(from_id, status, &channel_id, false);
                    update_redis_async(from_id.clone(), joined);
                    tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });

                    for user_id in refuse_users {
                        let (cancel_tx, mut cancel_rx) = mpsc::unbounded_channel::<()>();
                        refuse_procedure_map.insert(user_id.clone(), cancel_tx);
                        let event_sender_clone = event_sender.clone();
                        let from_id_clone = from_id.clone();
                        let channel_id_clone = channel_id.clone();
                        tokio::spawn(async move {
                            tokio::select! {
                                _ = sleep(Duration::from_secs(15)) => {
                                    println!("20秒内没有接听，准备挂断通话");
                                    let joined = update_user_status(&user_id, "idle", "", false);
                                    update_redis_async(user_id.clone(), joined);
                                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                                        send_inside_message(&sender, &event_sender_clone, serde_json::json!({
                                            "msgType": "CmdHangup",
                                            "fromID": "0",
                                            "fromName": "Server",
                                            "toID": user_id,
                                            "msgData": {"channelId": "", "rtcToken": ""}
                                        }).to_string(), &user_id).await;
                                    }
                                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == from_id_clone).map(|e| e.key().clone()) {
                                        send_inside_message(&sender, &event_sender_clone, serde_json::json!({
                                            "msgType": "CmdRefuse",
                                            "fromID": user_id,
                                            "fromName": "Unknown",
                                            "toID": from_id_clone,
                                            "msgData": {"channelId": "", "rtcToken": ""}
                                        }).to_string(), &from_id_clone).await;
                                    }
                                    let left_users = get_users_by_channel(&channel_id_clone);
                                    if left_users.len() <= 1 {
                                        if let Some(user_id) = left_users.get(0) {
                                            if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == *user_id).map(|e| e.key().clone()) {
                                                let joined = update_user_status(user_id, "idle", "", false);
                                                update_redis_async(user_id.clone(), joined);
                                                send_inside_message(&sender, &event_sender_clone, serde_json::json!({
                                                    "msgType": "CmdHangup",
                                                    "fromID": "0",
                                                    "fromName": "Server",
                                                    "toID": user_id,
                                                    "msgData": {}
                                                }).to_string(), user_id).await;
                                            }
                                        }
                                    }
                                    tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
                                }
                                _ = cancel_rx.recv() => println!("用户已接听或挂断 {}", user_id),
                            }
                        });
                    }
                }
            }
        }

        "Refuse" => {
            println!("用户拒接电话 {}", from_id);
            if let Some(sender) = refuse_procedure_map.get(from_id) {
                sender.send(()).ok();
            }
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                let channel_id = user_data_vec[CHANNEL_IDX].clone();
                let joined = update_user_status(from_id, "idle", "", false);
                update_redis_async(from_id.clone(), joined);
                send_inside_message(&target_sender_which, event_sender, serde_json::json!({
                    "msgType": "CmdHangup",
                    "fromID": "0",
                    "fromName": "Server",
                    "toID": from_id,
                    "msgData": {"channelId": "", "rtcToken": ""}
                }).to_string(), from_id).await;

                if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == client_message_data.to_id).map(|e| e.key().clone()) {
                    send_inside_message(&sender, event_sender, serde_json::json!({
                        "msgType": "CmdRefuse",
                        "fromID": from_id,
                        "fromName": "Server",
                        "toID": client_message_data.to_id,
                        "msgData": {"channelId": "", "rtcToken": ""}
                    }).to_string(), &client_message_data.to_id).await;
                } else {
                    println!("找不到toID对应的sender");
                }

                let chatters = get_users_by_channel(&channel_id);
                if chatters.len() == 1 {
                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == chatters[0]).map(|e| e.key().clone()) {
                        let joined = update_user_status(&chatters[0], "idle", "", false);
                        update_redis_async(chatters[0].clone(), joined);
                        send_inside_message(&sender, event_sender, serde_json::json!({
                            "msgType": "CmdCancelCall",
                            "fromID": "0",
                            "fromName": "Server",
                            "toID": chatters[0],
                            "msgData": {"channelId": "", "rtcToken": ""}
                        }).to_string(), &chatters[0]).await;
                    }
                }
                tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
            }
        }

        "EndMeeting" => {
            println!("step - EndMeeting - 1 收到挂断会议通知！");
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                if user_data_vec[HOST_IDX] != "1" {
                    send_error_message(&target_sender_which, event_sender, from_id, "只有主持人可以结束会议！").await;
                } else {
                    let channel_id = user_data_vec[CHANNEL_IDX].clone();
                    if !channel_id.is_empty() {
                        let users = get_users_by_channel(&channel_id);
                        for user_id in &users {
                            let joined = update_user_status(user_id, "idle", "", false);
                            update_redis_async(user_id.clone(), joined);
                            if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == *user_id).map(|e| e.key().clone()) {
                                send_inside_message(&sender, event_sender, serde_json::json!({
                                    "msgType": "CmdEndMeeting",
                                    "fromID": "0",
                                    "fromName": "Server",
                                    "toID": user_id,
                                    "msgData": {"channelId": "", "rtcToken": ""}
                                }).to_string(), user_id).await;
                            }
                        }
                        tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
                    }
                }
            }
        }

        "Hangup" => {
            println!("step - hangup -1 接收到hangup消息");
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                let channel_id = user_data_vec[CHANNEL_IDX].clone();
                let joined = update_user_status(from_id, "idle", "", false);
                update_redis_async(from_id.clone(), joined);
                send_inside_message(&target_sender_which, event_sender, serde_json::json!({
                    "msgType": "CmdHangup",
                    "fromID": "0",
                    "fromName": "Server",
                    "toID": from_id,
                    "msgData": {"channelId": "", "rtcToken": ""}
                }).to_string(), from_id).await;

                let remaining_users = get_users_by_channel(&channel_id);
                if remaining_users.is_empty() {
                    println!("当前频道没有人员，请重新发起通话");
                } else {
                    for user_id in &remaining_users {
                        if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == *user_id).map(|e| e.key().clone()) {
                            send_inside_message(&sender, event_sender, serde_json::json!({
                                "msgType": "CmdLeave",
                                "fromID": from_id,
                                "fromName": "Unknown",
                                "toID": user_id,
                                "msgData": {}
                            }).to_string(), user_id).await;
                        }
                    }
                    if remaining_users.len() == 1 {
                        // 判断时候已经存在挂断任务
                        let user_id = remaining_users[0].clone();
                        if !channel_hangup_procedure_map.contains_key(&channel_id) {
                            let (cancel_tx, cancel_rx) = mpsc::unbounded_channel::<()>();
                            channel_hangup_procedure_map.insert(channel_id.clone(), cancel_tx);
                            let event_sender_clone = event_sender.clone();
                            let channel_id_clone = channel_id.clone();
                            hangup_last_user(user_id, event_sender_clone, channel_id_clone, cancel_rx);
                        }
                    } else {
                        let (allowed, disallowed): (Vec<_>, Vec<_>) = remaining_users.into_iter().partition(|id| HOST_ENABLED_ID_SET.contains(id.as_str()));
                        let new_host = allowed.into_iter().next().or(disallowed.into_iter().next());
                        if let Some(host_id) = new_host {
                            let joined = update_user_status(&host_id, &ONLINE_USERS.get(&host_id).map(|v| v.split(',').next().unwrap_or("idle").to_string()).unwrap_or("idle".into()), &channel_id, true);
                            update_redis_async(host_id, joined);
                        }
                    }
                }
                tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
            }
        }

        "Connect" => {
            println!("收到客户端Connect消息连接 {} 频道信息 {:?}", from_id, client_message_data);
            let to_id = &client_message_data.to_id;
            if let Some(to_data) = ONLINE_USERS.get(to_id).map(|v| v.clone()) {
                let data_split = to_data.split(',').map(String::from).collect::<Vec<_>>();
                let channel_id = data_split[CHANNEL_IDX].clone();
                if channel_id.is_empty() {
                    send_error_message(&target_sender_which, event_sender, from_id, "对方数据出现异常，缺少channelID").await;
                    return;
                }

                if let Some(sender) = refuse_procedure_map.get(from_id) { sender.send(()).ok(); }
                if let Some(sender) = channel_hangup_procedure_map.get(&channel_id) { sender.send(()).ok(); }
                let mut real_host_id = String::new();
                if !ONLINE_USERS.iter().any(|e| e.value().split(',').nth(CHANNEL_IDX).unwrap_or("") == channel_id && e.value().split(',').nth(HOST_IDX).unwrap_or("") == "1") {
                    let initiator = channel_id.split('_').next().unwrap_or("");
                    let host_id = if HOST_ENABLED_ID_SET.contains(initiator) {
                        initiator.to_string()
                    } else {
                        ONLINE_USERS.iter()
                            .filter(|e| e.value().split(',').nth(CHANNEL_IDX).unwrap_or("") == channel_id)
                            .map(|e| e.key().clone())
                            .find(|id| HOST_ENABLED_ID_SET.contains(id.as_str()))
                            .unwrap_or_else(|| ONLINE_USERS.iter().find(|e| e.value().split(',').nth(CHANNEL_IDX).unwrap_or("") == channel_id).map(|e| e.key().clone()).unwrap_or_default())
                    };
                    real_host_id = host_id.to_string();
                    let joined = update_user_status(&host_id, "calling", &channel_id, true);
                    update_redis_async(host_id, joined);
                }

                let users = ONLINE_USERS.iter()
                    .filter(|e| e.value().split(',').nth(CHANNEL_IDX).unwrap_or("") == channel_id)
                    .map(|e| e.key().clone())
                    .collect::<Vec<_>>();
                for user_id in users {
                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                        send_inside_message(&sender, event_sender, serde_json::json!({
                            "msgType": "CmdConnect",
                            "msgData": {"channelID": channel_id, "rtcToken": ""},
                            "fromID": "0",
                            "fromName": "Server",
                            "toID": user_id
                        }).to_string(), &user_id).await;
                    }
                    if real_host_id != user_id {
                        let joined = update_user_status(
                            &user_id,
                            "calling",
                            &channel_id,
                            real_host_id == user_id
                        );
                        update_redis_async(user_id, joined);
                    }
                }
                tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
            } else {
                send_error_message(&target_sender_which, event_sender, from_id, "必须传递to_id").await;
            }
        }

        "KickOut" => {
            println!("step - KickOut - 1 收到踢出通知");
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                if user_data_vec[HOST_IDX] != "1" {
                    send_error_message(&target_sender_which, event_sender, from_id, "只有主持人才能踢出用户").await;
                } else {
                    let channel_id = user_data_vec[CHANNEL_IDX].clone();
                    let users = get_users_by_channel(&channel_id);
                    println!("step - KickOut - 2 获取到频道 {:?} 的所有在线用户 {:?}", channel_id, users);
                    // 存储最后一位的用户
                    let mut last_user_id = String::new();
                    for user_id in users.iter() {
                        if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == *user_id).map(|e| e.key().clone()) {
                            if *user_id == client_message_data.to_id {
                                let joined = update_user_status(&user_id, "idle", "", false);
                                update_redis_async(user_id.clone(), joined);
                                send_inside_message(&sender, event_sender, serde_json::json!({
                                    "msgType": "CmdHangup",
                                    "msgData": {"channelID": channel_id, "rtcToken": ""},
                                    "fromID": "0",
                                    "fromName": "Server",
                                    "toID": user_id
                                }).to_string(), &user_id).await;
                            } else {
                                last_user_id = user_id.clone();
                                send_inside_message(&sender, event_sender, serde_json::json!({
                                    "msgType": "CmdKickOut",
                                    "msgData": {"channelID": channel_id, "rtcToken": ""},
                                    "toID": &user_id,
                                    "fromName": "Server",
                                    "fromID": "0"
                                }).to_string(), &user_id).await;
                            }
                        }
                    }
                    // 判断是否只留下一人
                    println!("step - KickOut - 3 判断是否只留下一人 {}", users.len());
                    if users.len() <= 2 {
                        if !channel_hangup_procedure_map.contains_key(&channel_id) {
                            // 创建channel挂断
                            let (cancel_tx, cancel_rx) = mpsc::unbounded_channel::<()>();
                            channel_hangup_procedure_map.insert(channel_id.clone(), cancel_tx);
                            hangup_last_user(last_user_id, event_sender.clone(), channel_id, cancel_rx);
                        }
                    }
                    tokio::spawn(async move { notify_all_clients_to_update_online_users().await; });
                }
            }
        }

        "Mute" => {
            println!("step - Mute - 1 收到静音通知");
            let to_id = &client_message_data.to_id;
            if to_id.is_empty() {
                send_error_message(&target_sender_which, event_sender, from_id, "必须传递to_id").await;
                return;
            }
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                if user_data_vec[HOST_IDX] != "1" {
                    send_error_message(&target_sender_which, event_sender, from_id, "您不是主持人，无法静音他人").await;
                    return;
                }
                let channel_id = user_data_vec[CHANNEL_IDX].clone();
                let users = get_users_by_channel(&channel_id);
                let json_str = serde_json::json!({
                    "msgType": "CmdMute",
                    "msgData": {},
                    "fromID": to_id,
                    "fromName": "Server",
                    "toID": to_id
                }).to_string();
                for user_id in users {
                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                        send_inside_message(&sender, event_sender, json_str.clone(), &user_id).await;
                    }
                }
            }
        }

        "MuteAll" => {
            println!("step - MuteAll - 1 收到静音全部通知");
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                if user_data_vec[HOST_IDX] != "1" {
                    send_error_message(&target_sender_which, event_sender, from_id, "只有主持人才有操作权限").await;
                    return;
                }
                let channel_id = user_data_vec[CHANNEL_IDX].clone();
                let users = get_users_by_channel(&channel_id);
                for user_id in users {
                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                        send_inside_message(&sender, event_sender, serde_json::json!({
                            "msgType": "CmdMuteAll",
                            "msgData": {"channelID": channel_id, "rtcToken": ""},
                            "fromID": "0",
                            "fromName": "Server",
                            "toID": user_id
                        }).to_string(), &user_id).await;
                    }
                }
            }
        }

        "MuteSelf" | "UnMuteSelf" => {
            println!("step - MuteSelf-UN - 1 收到开关静音自己通知");
            if let Some(user_data) = ONLINE_USERS.get(from_id).map(|v| v.clone()) {
                let user_data_vec = user_data.split(',').map(String::from).collect::<Vec<_>>();
                let channel_id = user_data_vec[CHANNEL_IDX].clone();
                let users = get_users_by_channel(&channel_id);
                for user_id in users {
                    if let Some(sender) = CLIENT_SENDERS.iter().find(|e| e.key().0 == user_id).map(|e| e.key().clone()) {
                        send_inside_message(&sender, event_sender, serde_json::json!({
                            "msgType": client_message_data.msg_type.as_str(),
                            "msgData": {},
                            "fromID": from_id,
                            "fromName": "Unknown",
                            "toID": user_id
                        }).to_string(), &user_id).await;
                    }
                }
            }
        }

        _ => {}
    }
}