use std::collections::HashMap;
use dashmap::DashSet;
use lazy_static::lazy_static;
use tokio::net::TcpStream;
use tokio_tungstenite::{accept_hdr_async, WebSocketStream};
use tungstenite::handshake::client::Request;
use tungstenite::handshake::server::Response;

// 存储可以为host的id集合
// 存储hasMike以及hasCamera
lazy_static! {
    pub static ref HOST_ENABLED_ID_SET: DashSet<String> = DashSet::new();
}

// 提取出来的处理握手的函数
pub(crate) fn handle_handshake(
    req: &tungstenite::handshake::server::Request,
    must_existed_params: &[&str],
    static_ws_pwd: &str,
) -> Result<HashMap<String, String>, String> {
    println!("新客户端连接: {}", req.uri());
    let connection_params = match crate::utils::utils::get_connection_params(req.uri().to_string())
    {
        Ok(p) => p,
        Err(e) => {
            let error_msg = format!("缺少重要连接数据段: {}", e);
            println!("{}", error_msg);
            return Err(error_msg);
        }
    };

    println!("连接参数: {:?}", connection_params);

    let not_existed = must_existed_params
        .iter()
        .find(|param| !connection_params.contains_key(&param.to_string()));

    if let Some(param) = not_existed {
        let error_msg = format!("缺少重要连接数据段: {}", param);
        println!("{}", error_msg);
        return Err(error_msg);
    }

    if connection_params.get("wsPwd").unwrap() != static_ws_pwd {
        println!("wsPwd不正确！");
        return Err("wsPwd不正确！".to_string());
    }

    // 存储是否为host
    if connection_params.get("enable_host").is_some() {
        println!("当前id允许可以成为host！ {}", connection_params.get("fromID").unwrap());
        HOST_ENABLED_ID_SET.insert(connection_params.get("fromID").unwrap().to_string());
    }

    println!("握手成功！");

    Ok(connection_params)
}

pub async fn handle_websocket_handshake(
    stream: TcpStream,
    must_existed_params: &[&str],
    static_ws_pwd: &str,
    connection_params: &mut Option<HashMap<String, String>>
) -> Result<WebSocketStream<TcpStream>, ()> {

    let ws_stream = match accept_hdr_async(stream, |req: &Request, resp| {
        match handle_handshake(req, must_existed_params, static_ws_pwd) {
            Ok(params) => {
                *connection_params = Some(params);
                Ok(resp)
            }
            Err(error_msg) => {
                println!("{}", error_msg);
                let error_resp = Response::builder()
                    .status(400)
                    .header("Content-Type", "text/plain")
                    .body(Some(error_msg))
                    .unwrap();
                Err(error_resp)
            }
        }
    })
        .await
    {
        Ok(ws) => ws,
        Err(e) => {
            println!("WebSocket握手失败: {}", e);
            return Err(());
        }
    };

    Ok(ws_stream)
}
