use futures::SinkExt;
use crate::client::ONLINE_USERS;
use serde::{Deserialize, Serialize};
use tungstenite::{Error, Message};
use crate::utils::json_utils::get_current_timestamp;

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct OnlineUserMessage {
    #[serde(rename = "callState")]
    pub call_state: String,
    #[serde(rename = "channelID")]
    pub channel_id: String,
    #[serde(rename = "deviceID")]
    pub device_id: String,
    #[serde(rename = "fromID")]
    pub from_id: String,
    #[serde(rename = "hasCamera")]
    pub has_camera: String,
    #[serde(rename = "hasMike")]
    pub has_mike: String,
    #[serde(rename = "isHost")]
    pub is_host: String,
    #[serde(rename = "userCallGroup")]
    pub user_call_group: String,
    #[serde(rename = "fromName")]
    pub from_name: String,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct ServerOnlineUserMessage {
    #[serde(rename = "msgType")]
    pub msg_type: String,
    #[serde(rename = "fromId")]
    pub from_id: String,
    #[serde(rename = "fromName")]
    pub from_name: String,
    #[serde(rename = "toId")]
    pub to_id: String,
    #[serde(rename = "msgData")]
    pub msg_data: Vec<OnlineUserMessage>,
    #[serde(rename = "time")]
    pub time: u128,
}

pub fn online_messages(messages: Vec<OnlineUserMessage>, from_id: String) -> serde_json::Result<String> {
    serde_json::to_string(&ServerOnlineUserMessage {
        msg_type: "CmdUpdateUserList".to_string(),
        from_name: "Server".to_string(),
        from_id: "0".to_string(),
        to_id: from_id,
        msg_data: messages,
        time: get_current_timestamp(),
    })
}

pub async fn send_online_users_and_send(
    sender: &mut (impl SinkExt<Message, Error=Error> + Unpin),
    from_id: &str,
) -> Result<(), Error> {
    let messages = send_online_users_resp()
        .await
        .map_err(|e| Error::Io(std::io::Error::new(std::io::ErrorKind::Other, e)))?;
    let json_message = online_messages(messages, from_id.to_string())
        .map_err(|e| Error::Io(std::io::Error::new(std::io::ErrorKind::Other, e)))?;
    if let Err(e) = sender.send(Message::text(json_message)).await {
        println!("发送在线用户列表消息给用户 {} 失败: {}", from_id, e);
        return Err(e);
    }
    println!("发送在线用户列表消息给用户 {} 成功", from_id);
    Ok(())
}

pub async fn send_online_users_resp() -> Result<Vec<OnlineUserMessage>, serde_json::Error> {
    let mut msg_data = Vec::new();
    {
        for entry in ONLINE_USERS.iter() {
            let user_info_str: &String = entry.value();
            let parts: Vec<&str> = user_info_str.split(',').collect();
            if parts.len() == 9 {
                let user_msg = OnlineUserMessage {
                    call_state: parts[0].to_string(),
                    channel_id: parts[1].to_string(),
                    device_id: parts[2].to_string(),
                    from_id: parts[3].to_string(),
                    has_camera: parts[4].to_string(),
                    has_mike: parts[5].to_string(),
                    is_host: parts[6].to_string(),
                    user_call_group: parts[7].to_string(),
                    from_name: parts[8].to_string(),
                };
                msg_data.push(user_msg);
            }
        }
    }

    Ok(msg_data)
}
