use crate::config::config::REDIS_ADDR;
use lazy_static::lazy_static;
use redis::Client;
use redis::Commands;
use redis_pool::SingleRedisPool;

lazy_static! {
    static ref REDIS_POOL: SingleRedisPool = {
        let client = Client::open(REDIS_ADDR).expect("Failed to connect to Redis");
        redis_pool::RedisPool::new(client, 16, Some(512))
    };
}

// 从 Redis 的 onlineUsers 集合中移除当前用户的信息
pub async fn remove_this_connection(from_id: &str) -> Result<(), redis::RedisError> {

    let mut con = REDIS_POOL
        .get_connection()
        .map_err(|e| {
            println!("Failed to get Redis connection: {}", e);
            e
        })?;

    if let Err(e) = con.hdel::<&str, &str, ()>("onlineUsers", from_id) {
        println!("从 Redis 中的 onlineUsers 哈希表删除用户信息时出错: {}", e);
        return Err(e);
    }
    Ok(())
}

pub async fn update_client_redis_data(
    from_id: &str,
    data_str: String,
) -> Result<(), redis::RedisError> {
    // 修改对应用户redis数据
    let mut con = REDIS_POOL
        .get_connection()
        .map_err(|e| {
            println!("Failed to get Redis connection: {}", e);
            e
        })?;
    if let Err(e) = con.hset::<&str, &str, &str, ()>("onlineUsers", from_id, &data_str) {
        println!("修改 Redis 中的 onlineUsers 哈希表时出错: {}", e);
        return Err(e);
    }
    Ok(())
}

// 将当前用户的信息插入到 Redis 的 onlineUsers 集合中
pub async fn insert_this_connection(
    from_id: &str,
    user_info_str: String,
) -> Result<(), redis::RedisError> {

    let mut con = REDIS_POOL
        .get_connection()
        .map_err(|e| {
            println!("Failed to get Redis connection: {}", e);
            e
        })?;

    if let Err(e) = con.hset::<&str, &str, &str, ()>("onlineUsers", from_id, &user_info_str) {
        println!(
            "将用户信息添加到 Redis 中的 onlineUsers 哈希表时出错: {}",
            e
        );
        return Err(e);
    }
    Ok(())
}
