use crate::config::config::STATIC_ADDR;
use std::collections::HashMap;
use std::env;

pub(crate) fn get_connection_params(
    connection_url: String,
) -> Result<HashMap<String, String>, &'static str> {
    if let Some(query_part) = connection_url.split('?').nth(1) {
        let mut params_mapping = HashMap::new();
        for param in query_part.split('&') {
            if let Some((key, value)) = param.split_once('=') {
                params_mapping.insert(key.to_string(), value.to_string());
            } else {
                return Err("Invalid parameter format, missing '='");
            }
        }
        Ok(params_mapping)
    } else {
        Ok(HashMap::new())
    }
}

pub fn get_bind_var(argv_name: String, env_name: String, default_value: String) -> Result<String, Box<dyn std::error::Error>>{
    // 1. 处理命令行参数
    let mut args = env::args().skip(1);
    if let Some(pos) = args.position(|arg| arg == argv_name) {
        // 获取下一个参数值
        return args.nth(pos)
            .ok_or("argv requires a value".into())
            .and_then(|addr| {
                Ok(addr)
            });
    }

    // 2. 处理环境变量
    if let Ok(env_addr) = env::var(env_name) {
        return Ok(env_addr);
    }

    // 3. 返回默认值
    Ok(default_value)
}

// 获取地址的逻辑封装
pub fn get_addr() -> Result<String, Box<dyn std::error::Error>> {
    Ok(get_bind_var("--addr".to_string(), "RS_WS_ADDR".to_string(), STATIC_ADDR.to_string())?)
}